
## TO DO:
## replace cat() with file connections?
## generalize options/save as text object?
## write options for translating theta/phi/r into viewpoint;
##  allow variations in axis width, lighting (beyond TRUE/FALSE), etc. etc.
## expand color options
## mma.segment?

livejarURL <- "http://wwwvis.informatik.uni-stuttgart.de/~kraus/LiveGraphics3D/live.jar"
get.live.jar <- function(url=livejarURL,
                         target.dir=getwd()) {
    message("Currently downloading LiveGraphics3D, which is free only for non-commercial use",
          "See the documentation for LG3d, and/or the LiveGraphics3D web site, for more info")
    download.file(url,file.path(target.dir,"live.jar"))
}

## utility strings for sticking livegraphics into a web page
## (might want to generalize WIDTH/HEIGHT eventually)

LG.html.head <-
  c("<HTML>",
    "<body>",
    "<APPLET ARCHIVE=\"live.jar\" CODEBASE=\".\" CODE=\"Live.class\" WIDTH=500 HEIGHT=500 ALIGN=LEFT>",
    "<PARAM NAME=INPUT VALUE=\"")

LG.html.tail <-
  c("\">",
    "</APPLET>",
    "</BODY>",
    "</HTML>")

## utils:::globalVariables( c("cur.LG","cur.LG.type",
##                           "cur.LG.options","cur.LG.display"),
##                           "LG3d")
cur.LG <- cur.LG.type <- cur.LG.options <- cur.LG.display <- NULL

LG.open <- function(basename="LG",
                    type="html",
                    display=FALSE,
                    check.file=TRUE,
                    options=NULL,fn) {
  if (! (type %in% c("html","mma")))
    stop("unknown output type")
  if (missing(fn)) {
    ext <- type
    fn <- paste(basename,ext,sep=".")
    if ("." %in% strsplit(basename,""))
      warning(paste("basename contains period, using output file",fn))
  }
  if (check.file && file.exists(fn)) {
    cat("File (",fn,") exists!\n",sep="")
    ## FIXME: could be more user-proof (default=backup)
    cat("Overwrite (y/n) or backup (b)? ")
    ans <- readline("")
    if (ans=="n")
      stop("livegraphics cancelled")
    if (ans=="b")
      file.copy(fn,paste(fn,".bak",sep=""))
  }
  if (!file.create(fn))
    stop("can't create output file")
  assign("cur.LG",fn,inherits=TRUE)  ## set global filename
  assign("cur.LG.type",type,inherits=TRUE)  ## set global filename
  assign("cur.LG.options",options,inherits=TRUE)  ## set options
  assign("cur.LG.display",display,inherits=TRUE)  ## display in browser?
  if (type=="html")
    cat(LG.html.head,file=cur.LG,append=TRUE)
  mma.start(file=cur.LG)
}

LG.display <- function(fn=cur.LG,
                       check.live.jar=TRUE,browser = getOption("browser"))
{
    mach <- .Platform$OS.type
    if (is.null(browser)) {
        ## if (mach=="Unix")
        ## stop("Invalid browser name, check options(\"browser\").")
        ## else  if (mach=="Win32") {
        ##    browser <- "C:/PROGRA~1/INTERN~1/IEXPLORE.EXE"
        ##    warning("Using default IE location for browser")
        stop("need to set browser options")
    }
    if (check.live.jar) {
    if (!file.exists("live.jar")) {
        cat("You'll need live.jar in your working directory to display.\n")
        cat("live.jar is part of LiveGraphics3D, which is free for\n")
        cat("non-commercial use; you can use get.live.jar() to download it.\n")
        cat("Do you want to download live.jar with get.live.jar() now? ")
        ans <- readline("")
        if (ans=="y")
          get.live.jar()
      }
    }
  ## assuming output file is in working directory ...
  cat("displaying LiveGraphics in browser ...\n")
  if (mach!="windows") {
    url <- paste("file:",file.path(getwd(),fn),sep="")
    system(paste(browser, " -remote \"openURL(", url, ")\" 2>/dev/null || ", 
                 browser, " ", url, " &", sep = ""))
  }
  else {
    ## is this robust??  will it do remote-open?
    shell(file.path(getwd(),fn), wait=FALSE)
    ##    system(paste(browser,file.path(getwd(),fn)))
  }
}
  
LG.close <- function(fn=cur.LG,type=cur.LG.type,options=cur.LG.options,
                     display=cur.LG.display) {
  if (!exists("cur.LG"))
    stop("cur.LG doesn't exist: have you opened an LG file?")
  mma.end(file=cur.LG)
  do.call("mma.options",c(list(file=fn),options))
  if (type=="html")
    cat(LG.html.tail,file=cur.LG,append=TRUE)
  if (display)
    LG.display(fn)
  ## Can't do this any more: "cannot remove bindings from a locked environment"
  ## rm(cur.LG,inherits=TRUE)
  ## rm(cur.LG.type,inherits=TRUE)
  ## rm(cur.LG.options,inherits=TRUE)
  ## rm(cur.LG.display,inherits=TRUE)
}                   

mma.digits <- 3   ## default signif. digits for Mma objects

## begin Mathematica graphics object
mma.start <- function(file=cur.LG,...)
  cat("Graphics3D[{\n",file=file,append=TRUE,...)

## separator for Mathematica graphics objects
mma.sep <- function(file=cur.LG,...) {
  cat(",\n",file=file,append=TRUE,...)
}

## mathematica brace
mma.brace <- function(file=cur.LG,...) {
  cat("}",append=TRUE,file=file,...)
}

## end a graphics object
mma.end <- function(file=cur.LG,...) {
  mma.brace(file=file,...)  ## closing bracket for graphics object
  mma.sep(file=file,...)    ## comma before graphics defs
}

## create a triplet in Mathematica format
mma.triplet <- function(x,y,z,digits=mma.digits)
  paste("{",paste(signif(x,digits),signif(y,digits),signif(z,digits),
                  sep=","),"}",sep="")

## issue a mathematica graphics directive
mma.color <- function(col,file=cur.LG,digits=mma.digits) {
  cat(paste("RGBColor[",paste(signif(col2rgb(col)/255,digits),
                              collapse=","),"]",sep=""),
      file=file,append=TRUE)
}

mma.edge <- function(col=par("fg"),file=cur.LG) {
  cat("EdgeForm[",file=file,append=TRUE)
  if (!is.na(col))
    mma.color(col,file=file)
  cat("]",file=file,append=TRUE)
}

## output a gridded surface in Mma format
## add: rounding, thinning, colors
mma.persp <- function(x,y,z,
                      digits=mma.digits,color.type="NONE",
                      colors=topo.colors(100),thin=1,
                      file=cur.LG,border=par("fg"),...) {
  if (thin != 1) {  ## thin surface
    x <- x[seq(1,length(x),by=thin)]
    y <- y[seq(1,length(y),by=thin)]
    z <- z[seq(1,nrow(z),by=thin),seq(1,ncol(z),by=thin)]
  }
  zmin <- min(z)
  zmax <- max(z)
  zrng <- zmax-zmin
  mma.edge(border,file=file)
  mma.sep(file=file)
  ncolor <- length(colors)
  for (i in seq(1,length(x)-1)) {
    for (j in seq(1,length(y)-1)) {
      if (color.type=="Z") {  ## set color corresponding to height
        znorm <- (z[i,j]-zmin)/zrng
        col <- colors[ceiling(znorm*(ncolor-1)+1)]
        mma.color(col,file=file)
      }
      mma.polygon(matrix(c(x[i],y[j],z[i,j],
                           x[i+1],y[j],z[i+1,j],
                           x[i+1],y[j+1],z[i+1,j+1],
                           x[i],y[j+1],z[i,j+1]),
                         byrow=TRUE,ncol=3))
      if ((i+1)<length(x) || (j+1)<length(y))  ## separate rows/columns
        cat(",\n",append=TRUE,file=file,...)
    }
  }
}

## lots and lots of Mathematica graphics formatting code 
## might be interested in modifying:
## BoxRatios, Lighting, AxesLabel, FaceGrids, PlotRange, ...

mma.options <- function(lighting=TRUE,xlab="x",ylab="y",zlab="z",
                        boxrat=c(1,1,0.6),
                        plotlabel=NULL,
                        axes=TRUE,
                        box=TRUE,
                        file=cur.LG,...) {
  nopt <- 31
  optstr <- character(nopt)
  optstr[1] <- "SphericalRegion -> False"
  optstr[2] <- "PlotRange -> Automatic"
  optstr[3] <- "DisplayFunction -> (Display[$Display, #1] & )"
  optstr[4] <- "ColorOutput -> Automatic"
  optstr[5] <- paste("Axes ->",ifelse(axes,"True","False"))
  optstr[6] <- paste("PlotLabel -> ",ifelse(is.null(plotlabel),"None",
                                            plotlabel))
  optstr[7] <- paste("AxesLabel -> {``",xlab,
                     "``,``",ylab,
                     "``,``",zlab,"``}",sep="")
  optstr[8] <- "Ticks -> Automatic"
  optstr[9] <- "Prolog -> {}"
  optstr[10] <- "Epilog -> {}"
  optstr[11] <- "AxesStyle -> {Thickness[0.01]}"
  optstr[12] <- "Background -> Automatic"
  optstr[13] <- "DefaultColor -> Automatic"
  optstr[14] <- "DefaultFont -> {''Courier'', 10.}"
  optstr[15] <- "AspectRatio -> Automatic"
  optstr[16] <- "ViewPoint -> {1.3, -2.4, 2.}"
  optstr[17] <- ifelse(box,
                       "Boxed -> True",
                       "Boxed -> False")
  optstr[18] <- paste("BoxRatios -> {",paste(boxrat,collapse=","),"}",
                      sep="")
  optstr[19] <- "Plot3Matrix -> Automatic"
  optstr[20] <- "AmbientLight -> GrayLevel[0]"
  if (lighting) {
    optstr[21] <- paste("Lighting -> True",
                        paste("LightSources -> {",
                        "{{1., 0., 1.}, RGBColor[1, 0, 0]},",
                        "{{1., 1., 1.}, RGBColor[0, 1, 0]},",
                        "{{0., 1., 1.}, RGBColor[0, 0, 1]}}",sep=""),
                        sep=",\n")
  }
  else {
    optstr[21] <- "Lighting -> False"
  }
  optstr[22] <- "ViewCenter -> Automatic"
  optstr[23] <- "PlotRegion -> Automatic"
  optstr[24] <- "ImageSize -> Automatic"
  optstr[25] <- "TextStyle -> {FontFamily -> ''TimesRoman''}"
  optstr[26] <- "FormatType -> StandardForm"
  optstr[27] <- "ViewVertical -> {0., 0., 1.}"
  optstr[28] <- "FaceGrids -> None"
  optstr[29] <- "Shading -> True"
  optstr[30] <- "AxesEdge -> Automatic"
  optstr[31] <- "BoxStyle -> Automatic"
  optval <- paste("{",
                  paste(optstr,collapse=",\n"),
                  ## closing square brackets for Graphics3D[ from mma.start()
                  "}]\n")
  cat(optval,file=file,append=TRUE)
}

## output text as Mma graphics object
mma.text <- function(x,y,z,text,digits=mma.digits,col=par("fg"),file=cur.LG,...) {
  col <- rep(col,length(x))
  for (i in (1:length(x))) {
    mma.color(col[i],file=file)
    cat("Text[``",text[i],"``,",
        mma.triplet(x[i],y[i],z[i],digits=digits),"]",sep="",append=TRUE,file=file,...)
    if (i<length(x))
      cat(",",append=TRUE,file=file,...)
    cat("\n",append=TRUE,file=file,...)
  }
}

mma.polygon <- function(pmat,file=cur.LG,digits=mma.digits,...) {
  cat("Polygon[{",append=TRUE,file=file,...)
  cat(paste(apply(pmat,1,function(z)do.call("mma.triplet",
                                            as.list(c(z,digits=digits)))),
            collapse=","),
      file=file,append=TRUE)
  cat("}]",append=TRUE,file=file,...)
}

## output point as Mma graphics object
mma.point <- function(x,y,z,digits=mma.digits,col=par("fg"),pointsize=0.01,
                      file=cur.LG,...) {
  col <- rep(col,length(x))
  pointsize <- rep(pointsize,length(x))
  for (i in (1:length(x))) {
    mma.color(col[i],file=file)
    mma.sep(file=file)
    cat("PointSize[",pointsize[i],"],",sep="",append=TRUE,file=file,...)
    cat("Point[",
        mma.triplet(x[i],y[i],z[i],digits=digits),
        "]",sep="",append=TRUE,file=file,...)
    if (i<length(x))
      cat(",",append=TRUE,file=file,...)
    cat("\n",append=TRUE,file=file,...)
  }
}

## output line as Mma graphics object
mma.line <- function(x,y,z,digits=mma.digits,col=par("fg"),lwd=par("lwd"),
                     file=cur.LG,...) {
  mma.color(col,file=file)
  cat("Line[{\n",append=TRUE,...)
  for (i in (1:length(x))) {
    cat(mma.triplet(x[i],y[i],z[i],digits=digits),append=TRUE,file=file,...)
    if (i<length(x))
      cat(",",append=TRUE,file=file,...)
    cat("\n",append=TRUE,file=file,...)
  }
}

## example: output a likelihood surface with two lines
## (likelihood profiles in two
## directions) and the best-fit point in Mathematica format
LG.plot.profiles <- function(basename,x,y,surf,best,prof1,prof2,
                              xlab="x",ylab="y",zlab="z",
                     colors=heat.colors(100),thin=1,
                              profcol=c("red","green"),ptcol="blue",
                             fn=cur.LG) {
  LG.open(basename,options=list(xlab=xlab,ylab=ylab,zlab=zlab,lighting=FALSE))
  mma.persp(x,y,surf,color.type="Z",colors=colors,thin=thin)
  mma.sep()
  mma.line(prof1[,1],prof1[,2],prof1[,3],file=fn,col=profcol[1])
  mma.sep()
  mma.line(prof2[,1],prof2[,2],prof2[,3],file=fn,col=profcol[2])
  mma.sep()
  mma.point(best[1],best[2],best[3],col=ptcol,file=fn)
  LG.close()
}



