require(MASS)

covmat <- function(vars,cors,tol=1e-7) {
  n <- length(vars)
  cors <- rep(cors,length.out=n*(n-1)/2)
  m <- outer(sqrt(vars),sqrt(vars))
  ## is there a more efficient way to create a symmetric matrix?
  m[lower.tri(m)] <- m[lower.tri(m)]*cors
  m <- t(m)
  m[lower.tri(m)] <- m[lower.tri(m)]*cors
  if (any(eigen(m)$values<(-tol)))
    warning("non-positive definite matrix")
  m
}

strip.blanks <- function(str) {
 sub(' +$', '', sub('^ +', '', str))
}

skip.blank.lines <- function(con) {
  junk <- readLines(con,1)
  n <- 0
  while(nchar(junk)==0 || length(grep("[^ ]",junk))==0) {
    junk <- readLines(con,1)
    n <- n+1
  }
  pushBack(junk,con)
  n
}

phillips.getpmat <- function(con,n,hdrlines=1) {
  readLines(con,hdrlines) ## strip header
  lines <- readLines(con,n)
  s1 <- strsplit(lines,":")
  labs <- sapply(s1, function(z)strip.blanks(z[1]))
  vals <- t(sapply(sapply(s1,"[",2),
                   function(z)
                   as.numeric(strsplit(z,"[:(),=]")[[1]][c(1,3,5)])))
  dimnames(vals) <- list(labs,c("chi-square","df","p"))
  vals
}

phillips.getvec <- function(con,n=3,hdrlines=1) {
  readLines(con,hdrlines)
  vec = numeric()
  while(length(vec)<n) {
    readLines(con,1)
    vec = c(vec,scan(con,n=n, nlines=1,quiet = TRUE))
  }
  vec
}

phillips.getsubmat <- function(con,n=3) {
  readLines(con,1)
  mstr <- readLines(con,n=n)
  t(sapply(strsplit(mstr," +"),
         function(z)as.numeric(z[-(1:2)])))
}

phillips.getmat <- function(con,n=3,hdrlines=1) {
  readLines(con,hdrlines)
  m = matrix(nrow=n,ncol=0)
  while(ncol(m)<n) {
    m = cbind(m,phillips.getsubmat(con,n))
    skip.blank.lines(con)
  }
  m
}

phillips.getcolonval <- function(con) {
 str <- readLines(con,1)
 as.numeric(strsplit(str,":")[[1]][2])
}

run.cpc <- function(covs,npts,
                    progdir=NULL,
                    progname=NULL,
                    ansfn,datfn,
                    outfn,unlink.temp=TRUE) {
  if (is.null(progdir))
    progdir <- file.path(.path.package("cpc"),"exec")
  if (is.null(progname))
      progname=paste("phillips-cpc-",.Platform$OS.type,".exe",sep="")
  ngrp = length(covs)
  nvar = ncol(covs[[1]])
  npts = rep(npts,length.out=ngrp)
  outf <- file(description=datfn,open="w") ## open for  writing
  writeLines(c(paste("group",1:ngrp,sep="",collapse=" "),
               paste("var",1:nvar,sep="",collapse=" ")),con=outf)
  for (i in 1:ngrp) {
    writeLines(as.character(npts[i]),outf)
    write.table(covs[[i]],file=outf,
              row.names=FALSE,col.names=FALSE,quote=FALSE)
  }
  close(outf)
  progpath = file.path(progdir,progname)
  if (.Platform$OS.type=="unix") {
    ans <- file(description=ansfn,open="w") ## open for writing
    writeLines(c(datfn,outfn,"y","n","n",""),con=ans)
    close(ans)
    system(paste(progpath,"<",ansfn,"1> /dev/null 2>&1"))
  } else {  ## assume Windows!
    system(progpath,input=c(basename(datfn),basename(outfn),
                      "y","n","n",""))
  }
}

call.cpc <- function(data=NULL,cov1=NULL,cov2=NULL,
                     covs=NULL,npts=c(200,200),
                     progdir=NULL,
                     progname=NULL,
                     ansfn=NULL,
                     datfn=NULL,
                     outfn=NULL,
                     unlink.temp=TRUE,
                     use="complete.obs") {
  if (is.null(progdir))
    progdir = file.path(.path.package("cpc"),"exec")
  if (is.null(progname))
      progname=paste("phillips-cpc-",.Platform$OS.type,".exe",sep="")
  tmpdirf = function(x,base) {
    if(is.null(x)) switch(.Platform$OS.type,
                          unix=tempfile(base),
                          windows=tempfile(base,tmpdir=getwd())) else x
  }
  ansfn = tmpdirf(ansfn,"cpcans")
  datfn = tmpdirf(datfn,"cpcdat")
  outfn = tmpdirf(outfn,"cpcout")
  if (is.null(covs) && !is.null(cov1) && !is.null(cov2)) {
    covs = list(cov1,cov2)
  }
  if (is.null(covs) && !is.null(data)) {
    covs <- lapply(data,cov,use=use)
    npts <- sapply(data,nrow)
  }
  run.cpc(covs,npts,progdir,progname,ansfn,datfn,outfn,unlink.temp)
  ngrp = length(covs)
  nvar = ncol(covs[[1]])
  r = read.cpc(outfn,ngrp,nvar)
  if (unlink.temp) {
    unlink(ansfn)
    unlink(outfn)
    unlink(datfn)
  }
  return(r$evecs.CPC)
}

read.cpc <- function(outfn,ngrp,nvar) {
  con <- file(outfn,open="r")
  readLines(con,4)
  eqcrit <- phillips.getcolonval(con)
  eqpar <- phillips.getcolonval(con)
  skip.blank.lines(con)
  eqtestmat <- phillips.getpmat(con,n=nvar+1)
  skip.blank.lines(con)
  evals.pool <- phillips.getvec(con,n=nvar)
  skip.blank.lines(con)
  evecs.pool <- phillips.getmat(con,n=nvar)
  skip.blank.lines(con)
  cov.pool <- phillips.getmat(con,n=nvar)
  skip.blank.lines(con)
  ##Extract data for proportionality##
  readLines(con,1)
  propcrit <- phillips.getcolonval(con)
  proppar <- phillips.getcolonval(con)
  skip.blank.lines(con)
  proptestmat <- phillips.getpmat(con,n=nvar)
  skip.blank.lines(con)
  evals.prop <- phillips.getvec(con,n=nvar)
  skip.blank.lines(con)
  evecs.prop <- phillips.getmat(con,n=nvar)
  skip.blank.lines(con)
  cov.prop.grps <- lapply(1:ngrp,
                          function(x) {
                            readLines(con,1)  ## skip prop const
                            m <- phillips.getmat(con,n=nvar)
                            skip.blank.lines(con)
                            m
                          })
  ## extract data for Common Principal Components##
  readLines(con,1)
  CPCcrit <- phillips.getcolonval(con)
  CPCpar <- phillips.getcolonval(con)
  skip.blank.lines(con)
  CPCtestmat <- phillips.getpmat(con,nvar-1)
  skip.blank.lines(con)
  readLines(con,1)
  evals.cpc.grps <- lapply(1:ngrp,
                           function(x) {
                             m <- phillips.getvec(con,n=nvar)
                             skip.blank.lines(con)
                             m
                           })
  skip.blank.lines(con)
  evecs.CPC <- phillips.getmat(con,n=nvar)
  close(con)
  return(list(evecs.CPC=evecs.CPC))
}

#######################################################################
## Now do back projection

###mwm Changed Bug i.e. removed "length" from bpx<-t(length(bpmat%*%t(x))
## back-projection function: data x, principal component vector evec
## bmb reinstated length!!!!

bpmat <- function(evec) {
  return(diag(length(evec)) - evec %*% t(evec))
}

bpfun <- function(x,evec) {
  t(bpmat(evec) %*% t(x))
}

pooled.cpc <- function (x1, x2 = NULL, use="complete.obs") 
{
  if (!is.null(x2)) {
    X <- list(x1, x2)
  }
  else {
    X <- x1
  }
  Xsc <- do.call("rbind", lapply(X, scale, scale = FALSE))
  eigen(cov(Xsc,use=use))$vectors
}

######################################################################

## dat: data list (list of similar matrices, traits as columns,
##   rows as observations)
## ngrpn: number of observations per group (single number or vector)
## nvar: number of traits/variables
## N: total number of observations
## ngrp: number of groups
## cpcvecfun: function for calculating CPC
## calc.cov: if FALSE  calculate VARIANCES of all elements
##                     in eigenvector matrix
##           if TRUE calc. VAR-COV MATRIX of 1st column (CPC 1)
##   (first column of result when FALSE = diag of result when TRUE)
calc.cpcerr <- function(dat,
                       ngrpn=nrow(dat[[1]]),nvar=ncol(dat[[1]]),
                       N=NULL,ngrp=length(dat),
                       ##mwm replaced pooled.cpc with call.cpc
                       cpcmat=NULL,
                       cpcvecfun=call.cpc,
                       calc.cov=TRUE,
                        use="complete.obs") {
  ## bmb: changed default calc.cov to TRUE (!!)
  if (length(ngrpn)<ngrp) ngrpn <- rep(ngrpn,ngrp)
  if (is.null(N)) N <- sum(ngrpn)
  e <- lapply(dat,function(z) {eigen(cov(z,use=use))$values})
  theta.hat <- matrix(0,nrow=nvar,ncol=nvar)
  for (i in 1:ngrp) { ## fancy: uses "outer"
    theta.i = outer(e[[i]],e[[i]],
      function(x,y) { (x*y)/(x-y)^2})*(N/ngrpn[i])## bmb bug fix!
    theta.hat = theta.hat + 1/theta.i
  }
  theta.hat = 1/theta.hat ## calc. harmonic mean
  if (is.null(cpcmat))
    cpcmat <- cpcvecfun(dat) ## get common principal components
  ## fancy: sum with j!=h is equivalent to matrix mult. with
  ## diag(theta)==0
  diag(theta.hat) <- 0
  if (!calc.cov) {
    ## VARIANCES of all elements with themselves
    return(1/N*(cpcmat^2 %*% theta.hat))
  } else {
    ## VAR-COV matrix of eig. 1 only
    covarr <- matrix(0,nrow=nvar,ncol=nvar)
    for (j in 1:nvar) {
      covarr <- covarr + theta.hat[1,j]*(cpcmat[,j] %*% t(cpcmat[,j]))
    }
    return(covarr/N)
  }
}

#############################################################################
## calculate back-projection errors, based on errors of
## principal components

error.bp <- function(dat,
                     ngrpn=nrow(dat[[1]]),nvar=ncol(dat[[1]]),
                     N=NULL,ngrp=length(dat),
                     ##mwm replaced pooled.cpcvec with call.cpc
                     cpcmat=NULL,m=NULL,eigvar=NULL,
                     cpcvecfun=call.cpc,
                     use="complete.obs",
                     debug=FALSE) {
  if (!is.list(dat)) dat <- list(dat)
  if (is.null(cpcmat)) cpcmat <- cpcvecfun(dat,use=use)
  if (is.null(m)) m <- t(sapply(dat,apply,2,mean))
  totdev <- apply(abs(m),2,sum,na.rm=TRUE)
  ## find *total* deviation in each trait (ignore NAs)
  if (is.null(eigvar)) 
    eigvar <- calc.cpcerr(dat,ngrpn,nvar,N,ngrp,calc.cov=TRUE,
                          cpcmat=cpcmat,
                          cpcvecfun=cpcvecfun)
  bpvar <- matrix(nrow=nvar,ncol=nvar)
  cpcv1 <- cpcmat[,1] ## principal eigenvector
  for (i in 1:nvar) {
    for (j in 1:nvar) {
      ## fixed bug: square product
      ## added cov. term
      ## fixed bug: transposed cpcmat correctly
      bpvar[i,j] <- (cpcv1[i]*cpcv1[j])^2*
        (eigvar[i,i]/cpcv1[i]^2+eigvar[j,j]/cpcv1[j]^2)+
          2*cpcv1[i]*cpcv1[j]*eigvar[i,j]
    }
  }
  ##  if (return.bpvar) return(bpvar)
  bpcov = array(dim=c(nvar,nvar,nvar))
  ## (i,j,k) = covariance of b_ij with b_ik
  for (i in 1:nvar) {
    for (j in 1:(nvar-1)) {
      for (k in (j+1):nvar) {
        bpcov[i,j,k] = 2*cpcv1[i]*cpcv1[j]*eigvar[i,k]+
          2*cpcv1[i]*cpcv1[k]*eigvar[i,j]+
            cpcv1[i]^2*eigvar[j,k]+
              cpcv1[j]*cpcv1[k]*eigvar[i,i]
        ##        if (debug) cat(i,j,k,bpcov[i,j,k],"\n")
      }
    }
  }
  if (debug) print(bpcov)
  bperr <- bpvar %*% totdev^2
  for (i in 1:nvar) {
    for (j in 1:(nvar-1))
      for (k in (j+1):nvar) {
        covterm <- 2*totdev[j]*totdev[k]*bpcov[i,j,k]
        if (debug) cat(i,j,k,bperr,covterm,"\n")
        bperr[i] <- bperr[i]+covterm
      }
  }
  if (any(bperr<0)) warning("estimated BP variance <0:",min(bperr))
  if (debug) cat(dim(bperr),"\n",names(dat),"\n",colnames(dat[[1]]),"\n")
  indiv.errs <- FALSE
  if (indiv.errs) {
    ## obsolete/unused code from when this function returned
    ## est. errors for each group and variable: now returning
    ## aggregated errors by variable
    rownames(bperr) <- names(dat)
    colnames(bperr) <- colnames(dat[[1]])
  } else {
    rownames(bperr) <- colnames(dat[[1]])
  }
  bperr
}

error.bp2 <- function(...) {
  warning("error.bp2 is deprecated: use error.bp instead")
  error.bp(...)
}

#################################################################
## combined error from back-projection and within-group variation
## (variance of the mean)

## obsolete?

# error.comb <- function(dat,
#                        ngrpn=nrow(dat[[1]]),nvar=ncol(dat[[1]]),
#                        N=NULL,ngrp=length(dat),
#                        cpcvecfun=call.cpc) {
#   if (!is.list(dat)) dat <- list(dat)
#   bperr <- error.bp(dat,ngrpn,nvar,N,ngrp,cpcvecfun)
#   evec <- cpcvecfun(dat)[,1]
#   bpmat <- diag(length(evec)) - evec %*% t(evec)
#   ## calculate variance for each element of each group, find
#   ## sum(bp^2 * var) for each group: gives contribution of within-group
#   ## variation to the total variation
#   err2 <- t(bpmat^2 %*% sapply(dat,function(x){apply(x,2,var)/nrow(x)}))
#   bperr+err2
# }

############################################################################
##ANOVA.Back projected

bp.anova <- function(x1,x2=NULL,
                     verbose=FALSE,cpcvecfun=call.cpc,
                     use="complete.obs",
                     debug=FALSE) {
  if (!is.null(x2)) x1 <- list(x1,x2)
  ## repackage data as a list of comparisons
  if (var(lapply(x1,ncol))>0)
    stop("groups don't all have same number of traits")
  nvar <- ncol(x1[[1]])
  ngrpn <- sapply(x1,nrow)
  cpc.evecs = cpcvecfun(x1)
  bpssq <- error.bp(x1,cpcmat=cpc.evecs,cpcvecfun=cpcvecfun,
                    use=use,debug=debug)*ngrpn^2
  e11 = cpc.evecs[,1]
  bpx = lapply(x1,bpfun,e11)
  N <- sum(ngrpn)
  X <- matrix(ncol=nvar,nrow=N)
  alist <- list()
  f <- factor(rep(1:length(x1),ngrpn))
  for (i in 1:nvar) {
    ## flatten back-projected variables from different groups
    X[,i] <- do.call("c",lapply(bpx,function(z)z[,i]))
    a <- anova(lm(X[,i] ~ f))
    ## add bp error to sum of squares
    a$"Sum Sq"[2] <- a$"Sum Sq"[2] + bpssq[i]
    a$"Mean Sq"[2] <- a$"Sum Sq"[2]/a$"Df"[2]
    a$"F value"[1] <- a$"Mean Sq"[1]/a$"Mean Sq"[2]
    a$"Pr(>F)"[1] <- pf(a$"F value"[1],
                        a$"Df"[1],
                        a$"Df"[2],lower.tail=FALSE)
    alist[[i]] <- a
  }
  names(alist) <- colnames(x1[[1]])
  if (!verbose) alist else
  list(alist=alist,bp=bpx,cpc.evecs=cpc.evecs)
}

plot.multigrp = function(x,vars=1:ncol(x[[1]]),
     cols=1:ncol(x[[1]]),...) {
   ngrp=length(x)
   npts = sapply(x,nrow)
   grps = rep(1:ngrp,npts)
   alldat = do.call("rbind",x)[,vars]
   ranges = apply(do.call("rbind",x),2,range)
   pairs(alldat,col=cols[grps],...)
}

simdata <- function(vars=c(10,10,10),cors=.8,npts=200,
                    seed=NULL,m1a=rep(0,length(vars)),
                    offset=1, offset2=0) {
  ndim <- length(vars)
  VC0 <- covmat(vars,cors)
  e0 <- eigen(VC0)
  m2a <- m1a+(offset*e0$vectors[,1])+(offset2*e0$vectors[,2])
  if (!is.null(seed)) set.seed(seed)
  list(mvrnorm(npts,mu=m1a,Sigma=VC0),
       mvrnorm(npts,mu=m2a,Sigma=VC0))
}

meancorrect <- function(data) {
  meanvals <- apply(do.call("rbind",data),2,mean)
  lapply(data,scale,scale=FALSE,center=meanvals)
}

## calculate cover
coverfun <- function(p,alpha=0.05,one.tailed=FALSE,na.rm=TRUE) {
  if (na.rm) p <- p[!is.na(p)]
  if (!one.tailed) {
    return(sum(p<alpha/2 | p>(1-alpha/2))/length(p))
  } else {
    return(sum(p<alpha)/length(p))
  }
}

