#include <stdlib.h>
#include <stdio.h>
#include "local.h"

#define MAX_VEC 3       // Maximum number of vector components;
#define MAX_DIM 10      // Maximum number of dimensions
#define MAX_BUFFER 256  // Max buffer size
#define NFILTERS   18
#define ISSTD      0    // Standard/Nonstandard wavelet basis (wvlt library)



extern int RecurFilter_(double **DATA, double **WORK, int* size, char* Wavelet, int* dim, int* iscoh, int vec, int verbose);
double Besov_Norm( double* data, int r, int p, int q, int dim, int* size );
extern int power(int num, int exp);
int intlog2( int x );

int main(int argc, char **argv)
{
int i, j, k, data_type, filter_alg, Ndim, Nelem, Nelem2, nf, coh, verbose, vec = 0;
int dimensions[MAX_DIM], rmin, rmax, pmin, pmax, qmin, qmax;
float *data4[MAX_VEC];
double *data8[MAX_VEC], *work[MAX_VEC], *work2[MAX_VEC], *store[MAX_VEC];
double besov;
FILE *fin, *fout1, *fout2;
waveletfilter *pftr;
char buffer[MAX_BUFFER], buffer2[MAX_BUFFER], filter[MAX_BUFFER]; 
char ext_dwt[20], ext_coh[20], ext_incoh[20];
char* filters_available[NFILTERS] = { "AdelsonSimoncelliHingorani",
				      "AntoniniBarlaudMatheiuDaubechies_4_4",     
				      "BattleLemarie",     
				      "BurtAdelson",    
				      "Coifman_12", // (Coifman wavelet with 12 coefficients i.e. 4 vanishing moments) 
				      "Daubechies_4",
				      "Daubechies_6",
				      "Daubechies_8",     
				      "Daubechies_10",     
				      "Daubechies_12",
				      "Daubechies_20",     
				      "Haar",     
				      "Pseudocoiflet_4_4",     
				      "Spline_2_2",     
				      "Spline_2_4",
				      "Spline_3_3",
				      "Spline_3_5",
				      "Spline_3_7" };


// -----------------------------------------------
//
//     $$ I. INITIALIZATION
//
// -----------------------------------------------


//         $$ I. 1.  CHECKING NUMBER OF PARAMETER FILES
//_____________________________________________________________________________________
 
 if ( argc < 2 )
 {
   perror("  Input file missing !\n\n");
   exit(EXIT_FAILURE);    
 }

//         $$ I. 2.  INITIALIZING ARRAYS
//_____________________________________________________________________________________

 for(i=0;i<MAX_DIM;i++)
   dimensions[i] = 1;

 for(k=0;k<MAX_VEC;k++)
 {
   data4[k] = NULL;
   data8[k] = NULL;
   work[k]  = NULL;
   work2[k] = NULL;
   store[k] = NULL;
 }

//         $$ I. 3.  READING VERBOSITY
//_____________________________________________________________________________________

// Reading the parameter file;
// Verbosity level;
  fscanf(stdin,"%d", &verbose);
  fgets(buffer, MAX_BUFFER, stdin);


//         $$ I. 4.  READING FIELD TYPE
//_____________________________________________________________________________________

// Scalar or vector field;
  fscanf(stdin,"%d", &vec);
  fgets(buffer, MAX_BUFFER, stdin);
  if ( vec <= 0 )
  {
    perror("  Problems determining whether vector / scalar !\n\n");
    exit(EXIT_FAILURE);    
  }    
  if ( verbose )
  {
    if ( vec == 1 )
      fprintf(stderr, "  Analyzing a scalar field \n");
    else
      fprintf(stderr, "  Analyzing a vector field with %d components \n", vec);      
  }

//         $$ I. 5.  READING DIMENSION OF THE FIELD
//_____________________________________________________________________________________

// Dimension of the field;
  fscanf(stdin,"%d", &Ndim);
  fgets(buffer, MAX_BUFFER, stdin);
  if ( Ndim < 3 )
  {
    if ( verbose )
      fprintf(stderr, "  Dimension= %d \n", Ndim);
  }
  else
  {
    perror("  Dimension must be less than 4 !\n\n");
    exit(EXIT_FAILURE);    
  }

//         $$ I. 6.  READING SIZE OF THE FIELD
//_____________________________________________________________________________________

// Size of the field;
  Nelem = 1;
  Nelem2 = 1;
  if ( verbose )  
    fprintf(stderr, "  Size:  ");
  for(i=0;i<Ndim;i++)
    if ( fscanf(stdin, "%d", &dimensions[i]) != 1 )
    {
      perror("  Problems reading dimensions !\n\n");
      exit(EXIT_FAILURE);    
    }
    else
    {
      if ( verbose )
	fprintf(stderr," %d ", dimensions[i]);
      Nelem  *= dimensions[i];
      Nelem2 *= dimensions[i] / 2;
    }
  fgets(buffer, MAX_BUFFER, stdin);
  if ( verbose )
    fprintf(stderr,"\n");


//         $$ I. 7.  READING DATA FORMAT
//_____________________________________________________________________________________

// Bytes per number;
  fscanf(stdin,"%d", &data_type);
  switch (data_type)
  {
    case 0:
      if ( verbose )
	fprintf(stderr,"  Reading ASCII numbers ... \n");
      break;      
    case 4:
      if ( verbose )	  
	fprintf(stderr,"  Reading floats ... \n");
      break;
    case 8:
      if ( verbose )
	fprintf(stderr,"  Reading doubles ... \n");
      break;   
    default:
      perror("  Wrong data tyle (size) !\n\n");
      exit(EXIT_FAILURE);    
      break;
  }
  fgets(buffer, MAX_BUFFER, stdin);


//         $$ I. 8.  READING FILTER TYPE
//_____________________________________________________________________________________

// Determining the filter;
  fscanf(stdin,"%s", &filter);
  i = 0;
  while ( i < NFILTERS && strcmp( filter, filters_available[i] ) != 0 ) i++;
  if ( i < NFILTERS )
  {
    if ( verbose )
      fprintf(stderr,"  Filter used: %s [%d] \n", filter, i);
  }
  else
  {
    perror("  Invalid filter !\n\n");
    exit(EXIT_FAILURE);    
  }
  fgets(buffer, MAX_BUFFER, stdin);


//         $$ I. 9.  READING FILTERING ALGORITHM
//_____________________________________________________________________________________

// Determining the filtering algorithm;
  fscanf(stdin,"%d", &filter_alg);
  switch (filter_alg)
  {
    case 0:
      if ( verbose )
	fprintf(stderr,"  Performing DWT only \n");
      break;      
    case 1:
      if ( verbose )
	fprintf(stderr,"  Using MAD filtering algorithm \n");
      break;      
    case 2:
      if ( verbose )
	fprintf(stderr,"  Using the recursive filtering algorithm \n");
      break;
    default:
      perror("  Invalid filtering algorithm !\n\n");
      exit(EXIT_FAILURE);    
      break;
  }
  fgets(buffer, MAX_BUFFER, stdin);


//         $$ I. 10.  READING FILE EXTENSIONS
//_____________________________________________________________________________________

  fscanf(stdin,"%s%s%s", ext_dwt, ext_coh, ext_incoh );
  if ( verbose )
    fprintf(stderr,"  Using the extensions: %s %s %s \n", ext_dwt, ext_coh, ext_incoh );
  fgets(buffer, MAX_BUFFER, stdin);


//         $$ I. 11.  READING INDEX BOUNDS FOR BESOV NORMS
//_____________________________________________________________________________________

// Reading lower bounds;
  fscanf(stdin,"%d%d%d", &rmin, &pmin, &qmin);
  fgets(buffer, MAX_BUFFER, stdin);
// Reading lower bounds;
  fscanf(stdin,"%d%d%d", &rmax, &pmax, &qmax);
  fgets(buffer, MAX_BUFFER, stdin);

  if ( verbose && filter_alg == 0 )
    fprintf(stderr,"  Will compute Besov norms B^%d_(%d,%d) - B^%d_(%d,%d) \n",
	    rmin, pmin, qmin, rmax, pmax, qmax );


//         $$ I. 10.  ALLOCATING MEMORY
//_____________________________________________________________________________________

// Allocating the memory;
  for(k=0;k<vec;k++)
  {
    work[k]  = malloc(Nelem*sizeof(double));
    data8[k] = malloc(Nelem*sizeof(double));

// for floats will need an additional array;
    if ( data_type == 4 )
      data4[k] = malloc(Nelem*data_type);
    if ( filter_alg == 1 )
      work2[k] = malloc(Nelem*sizeof(double));
    
// additional array will be needed for filtering;
    if ( filter_alg > 0 )
      store[k] = malloc(Nelem*sizeof(double));
  }
    

// -----------------------------------------------
//
//     $$ II. ANALYSIS
//
// -----------------------------------------------


//         $$ II. 1.  READING DATA
//_____________________________________________________________________________________


// Reading the input data;
  switch (data_type)
  {
    case 0:
// ASCII input;
      for(k=0;k<vec;k++)
      {
	fin = fopen( argv[1+k], "rt");     
	i = 0;
	while( EOF != fscanf(fin,"%lf", &data8[k][i++] ) );      
	if ( Nelem != (i - 1) )
	{
	  perror("  Incorrect data input ! \n\n");
	  exit(EXIT_FAILURE);
	}
	fclose(fin);
      }
      break;

    case 4:
      for(k=0;k<vec;k++)
      {
	fin = fopen( argv[1+k], "rb");     
	if ( Nelem != fread( data4[k], 4, Nelem, fin) )
	{
	  perror("  Incorrect data input ! \n\n");
	  exit(EXIT_FAILURE);
	}
	fclose(fin);
// Casting on double
	for(i=0;i<Nelem;i++)
	  data8[k][i] = data4[k][i];
      }
      break;

    case 8:
      for(k=0;k<vec;k++)
      {
	fin = fopen( argv[1+k], "rb");  
	if ( Nelem != fread( data8[k], 8, Nelem, fin) )
        {
	  perror("  Incorrect data input ! \n\n");
	  exit(EXIT_FAILURE);
	}
	fclose(fin);
      }
      break;
  }

//         $$ II. 2.  FILTERING
//_____________________________________________________________________________________


// Performing the filtering;
  switch(filter_alg)
  {
//             $$ II. 2. 1. DISCRETE WAVELET TRANSFORM
//_____________________________________________________________________________________
    case 0:  
      if ( (pftr = wfltr_select(filter, &nf)) == NULL )
      {
	fprintf(stderr,"  Error selecting filter: %d (%s) !\n\n", nf, filter);
	exit(nf);
      }      
      for(k=0;k<vec;k++)
	wxfrm_dand( data8[k], dimensions, Ndim, 1, ISSTD, pftr, work[k] );
// Opening a file for writing;
      fout2 = fopen("besov_norm.dat","wt");
// Calculating the Besov norms (works for scalar fields for the moment);
      for(i=rmin;i<=rmax;i++)
	for(j=pmin;j<=pmax;j++)
	  for(k=qmin;k<=qmax;k++)
	  {
	    besov = Besov_Norm( work[0], i, j, k, Ndim, dimensions );
	    if ( verbose )
	      fprintf(stderr,"  Besov norm B^%d_(%d,%d) =  %lf \n", i, j, k, besov);
	    fprintf(fout2," %2d  %2d  %2d    %lf  \n", i, j, k, besov);
	  }
      fclose(fout2);
      break;
//             $$ II. 2. 2. MAD FILTERING
//_____________________________________________________________________________________
    case 1:
// This case is unfortunately incomplete;
// storing the original field;
      for(k=0;k<vec;k++)
	for(i=0;i<Nelem;i++) 
	  store[k][i] = data8[k][i];       
      coh = 1; 
      MedianWltFilter_( data8, work, work2, dimensions, filter, &Ndim, &coh, &Nelem2); 
// Calculating the incoherent field;
      for(k=0;k<vec;k++)
	for(i=0;i<Nelem;i++)
	  work[k][i] = store[k][i] - data8[k][i];
      break;
//             $$ II. 2. 3. RECURSIVE FILTERING
//_____________________________________________________________________________________
    case 2:
      coh = 1;
// storing the original field;
      for(k=0;k<vec;k++)
	for(i=0;i<Nelem;i++)
	  store[k][i] = data8[k][i];      
      RecurFilter_(data8, work, dimensions, filter, &Ndim, &coh, vec, verbose);
// Calculating the incoherent field;
      for(k=0;k<vec;k++)
	for(i=0;i<Nelem;i++)
	  work[k][i] = store[k][i] - data8[k][i];
      break;
    }


//         $$ II. 3.  SAVING DATA
//_____________________________________________________________________________________

// Saving data;  
  for(k=0;k<vec;k++)
  {
    if ( filter_alg == 0 )
    {
      sprintf(buffer,"%s.%s", argv[1+k], ext_dwt);
      if ( verbose )
	fprintf(stderr,"  Saving file %s \n", buffer);
      if ( data_type == 0 )
      {
// ASCII output;
	fout1 = fopen(buffer,"wt");
// for the moment it works only in 1D and 2D;	
	for(i=0;i<dimensions[0];i++)
	{
	  for(j=0;j<dimensions[1];j++)
	    fprintf(fout1," % 14lf", work[k][i*dimensions[1]+j]);
	  fprintf(fout1,"\n");
	}
      }
      else
      {
 // binary (double) output;
	fout1 = fopen(buffer,"wb");
	fwrite( work[k], sizeof(double), Nelem, fout1);      
      }
      fclose(fout1);
    }
    else
    {
      sprintf(buffer ,"%s.%s", argv[1+k], ext_coh);
      sprintf(buffer2,"%s.%s", argv[1+k], ext_incoh);
      if ( verbose )
	fprintf(stderr,"  Saving files %s %s \n", buffer, buffer2);
      if ( data_type == 0 )
      {
// ASCII output;
	fout1 = fopen(buffer ,"wt");
	fout2 = fopen(buffer2,"wt");
// for the moment it works only in 1D and 2D;
	for(i=0;i<dimensions[0];i++)
	{
	  for(j=0;j<dimensions[1];j++)
	  {
	    fprintf(fout1," % 14lf", data8[k][i*dimensions[1]+j]);
	    fprintf(fout2," % 14lf",  work[k][i*dimensions[1]+j]);
	  }
	  fprintf(fout1,"\n");
	  fprintf(fout2,"\n");
	}
      }
      else
      {
// binary (double) output;
	fout1 = fopen(buffer ,"wb");
	fout2 = fopen(buffer2,"wb");
	fwrite(data8[k], sizeof(double), Nelem, fout1);      
	fwrite( work[k], sizeof(double), Nelem, fout2);            
      }
      fclose(fout1); 
      fclose(fout2);
    }
  }
  
  exit(EXIT_SUCCESS);
}


/////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
//     $$ FUNCTION double Besov_Norm( double* data, int r, int p, int q, int dim, int* size )
// The function calculates the Besov norm of the field passed as parameter. The Besov
// norm is calculated based on the wavelet representation;
// Parameters:
// data    - input data (in wavelet representation);
//  r      - the regularity parameter;
//  p      - exponent "in space"
//  q      - exponent "in scale"
// dim     - dimension of the field;
// size    - size of the field;
// WARNING - The function hasn't been tested for the case when size[0] != size[1]
/////////////////////////////////////////////////////////////////////////////////////////

double Besov_Norm( double* data, int r, int p, int q, int dim, int* size )
{
int j, j0, j1, k, l, shift, Jmax[MAX_DIM];
double brpq, sum_p;

// Determining the number of levels;
 for(k=0;k<MAX_DIM;k++)
   Jmax[k] = intlog2( size[k] ) - 1;

 brpq  = 0.0;
 shift = 1;

// Calculating the norm;
 for(j=0;j<=Jmax[0];j++)
 {
   sum_p = 0.0;                                                                 
   j0 = power(2,j);
   j1 = ( j <= Jmax[1] ) ? j0 : 1;
   for(l=0;l<j0;l++)
   {
     for(k=0;k<j1;k++)     
       sum_p += pow( abs(data[ l * size[1] + shift + k]), p );
     if ( dim > 1 )
       for(k=0;k<2*j1;k++)  
	 sum_p += pow( abs(data[ (shift + l) * size[1] + k]), p );	 
   }
   brpq += pow( 2.0, j * q * (0.5 + r - 1.0 / (double)p) ) * 
                           pow(sum_p, (double) q / (double) p);
   shift += j0;
 }

// Adding the scale function coefficient (zeroth element);
 brpq = abs(*data) + pow( brpq, 1.0 / (double)q );

 return brpq;
}


/////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
//     $$ FUNCTION int intlog2( int x )
// The function calculates the base-2 integer logarithm
// Parameters:
//  x      - the argument;
/////////////////////////////////////////////////////////////////////////////////////////

int intlog2( int x )
{
int l = 0;

 if ( x%2 != 0 && x != 1 )
 {
   perror("  intlog2: the argument is not an integer power of 2!!! \n\n");
   exit(EXIT_FAILURE);    
 }

  while ( x > 1 )
  {
    x /= 2;
    l++;
  }

  return l;
}
