#include "macro_definitions.i"


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!              Program TURB2D 
!
!  Pseudo-spectral simulation of 2D turbulence in the compressible 
!  (Shallow Water) and incompressible (Navier-Stokes) regime in 
!  periodic and wall-bounded domains.
!______________________________________________________________________

PROGRAM TURB2D 

!-----------------------------------------------
!
! $$ I. INITIALIZATION
!
!-----------------------------------------------

!     $$ I. 1. INCORPORATION OF MODULES
!__________________________________________________

  USE database
  USE parameters
  USE diagnostics

  IMPLICIT NONE

!     $$ I. 2. DECLARATION OF LOCAL VARIABLES
!__________________________________________________
  INTEGER :: K, H, I, N, SYNCHRO = 0
  REAL(KIND=SIZE_OF_REAL) :: TOP_INI, TOP_FIN, DAY, TTIME
  

!     $$ I. 3. INITIALIZATION OF PARAMETERS AND VARIABLES 
!__________________________________________________
  
  CALL INIT_PARAMETERS
  CALL INIT_DATA
  CALL INIT_FFT
  CALL INIT_FLOW

!-----------------------------------------------
!
! $$ II. EVOLUTION
!
!-----------------------------------------------
!
!     Evolution of the spectral fields
!     __________________________________________
!     The Shallow Water System:
!     DP/DT+D/DX(P*U)+D/DY(P*V)=0
!     DU/DT-(DV/DX-DU/DY+CORIOL)*V+D/DX(P+(U2+V2)/2)=0
!     DV/DT+(DV/DX-DU/DY+CORIOL)*U+D/DY(P+(U2+V2)/2)=0
!
!     To advance this system using leapfrog we need:
!     - to calculate 5 fields    PU,PV,PP,PO,PF
!     - to save 6 fields         PUA ,PVA ,PPA  and  PUAA, PVAA, PPAA
!     which in total gives 11 fields with dimensions (IDIM4I,IDIM4J) in core memory
!     9 FFT call and 5 products are required per time step
!
!     Integrated fiels:
!     PU    horizontal velocity
!     PV    vertical velocity
!     PP    pressure / geopotential velocity
!
!     Auxilliary fields:
!     PF   streamfunction
!     PO   vorticity


!     $$ II. 1. SETTING THE WALL CLOCK
!___________________________________________________

#ifdef CRAY
  TOP_INI=SECOND()                 
#else
#ifdef SX
  CALL CLOCK(TOP_INI)
#else
  CALL CPU_Time(TOP_INI)
#endif
#endif 

!     $$ II. 2. UPDATING THE TIME STEP
!___________________________________________________

  DO N = 1, NM 
     NSTEP  = NINIT + N 
     NSTEPA = NSTEP - 1 
     TIME   = TIME0 + (N - 1)*DT 
     TIMEA  = TIME - DT 
     DAY    = TIME / (24.*3600.)

!     $$ II. 3. SAVING FIELDS FOR THE NEXT TIME STEP
!___________________________________________________

! Which fields are saved depends on the integration scheme used;

    IF ( INCOMP /= 2 ) THEN
!$OMP PARALLEL DO
       DO I=1,IDIM4J
          PUAA(1:IDIM4I,I) = PUA(1:IDIM4I,I) 
          PVAA(1:IDIM4I,I) = PVA(1:IDIM4I,I) 
          PPAA(1:IDIM4I,I) = PPA(1:IDIM4I,I) 
       ENDDO
    ENDIF
!$OMP PARALLEL DO
    DO I=1,IDIM4J
       PUA(1:IDIM4I,I) = PU(1:IDIM4I,I) 
       PVA(1:IDIM4I,I) = PV(1:IDIM4I,I) 
       PPA(1:IDIM4I,I) = PP(1:IDIM4I,I) 
    ENDDO

! Checking if will need to do an AB step instead of leapfrog
    IF ( ISYNC == 1 .AND. N == 1 .AND. IINIT == 1 ) SYNCHRO = 1


!     $$ II. 4. TEMPORARILY REMOVING MEAN PRESSURE
!______________________________________________________
!
     MEAN_PRESS = PP(1,1) ! done to mitigate truncation errors due to large numbers
     PP(1,1)   = 0.0 

!     $$ II. 5. MEAN VELOCITY FORCING
!______________________________________
            
     IF ( FORC_VMEAN == 1 ) THEN
        PVAA(1,1) = U0
        PVA(1,1)  = U0
     ENDIF

!     $$ II. 6. FORCING PRESSURE FIELD
!_______________________________________________

     IF (IFORCE == 1) THEN 
        PP(2*KEXCIT+1,2*KEXCIT+1) = PMFORCE / 4.0
        PP(2*KEXCIT+1,2*KEXCIT+2) = 0.0 
        PP(2*KEXCIT+2,2*KEXCIT+1) = PMFORCE / 4.0
        PP(2*KEXCIT+2,2*KEXCIT+2) = 0.0 
     ENDIF

!     $$ II. 7. CALCULATING VORTICITY
!___________________________________________
!CDIR IEXPAND      
     CALL OROT (PU, PV, PO)       
     PO(1,1) = PO(1,1) + CORIOL 
 
!     $$ II. 8. UPDATING MASKS FOR THE WALL-BOUNDED CASE
!_____________________________________________________________________________________
     
     IF (PENALIZATION /=0) THEN
        WORK(1:IDIM4I,1:IDIM4J,1) = KSI(1:IDIM4I,1:IDIM4J)
        CALL INIPENA(PENALIZATION,NSTEP)
        DRAG(N,2) = ( 4 / ( PI * DIAMETER**2 * EPSILON ) )*SUM(PUAA(1:IDIM3I,1:IDIM3J) * KSI(1:IDIM3I,1:IDIM3J))
        DRAG(N,1) = -( 4 / ( PI * DIAMETER**2 * EPSILON ) )*SUM(PVAA(1:IDIM3I,1:IDIM3J) * KSI(1:IDIM3I,1:IDIM3J)) 

        WRITE(NFCX) DRAG(N,1)
        WRITE(NFCY) DRAG(N,2)
        IF (FLOW_CASE == 3) THEN
           VELO_FORCE = -KSI*U0/EPSILON
           KSI = EXP(-KSI*DT/EPSILON)
           IF (WALLS /= 0) WALL_MASK = EXP(-WALL_MASK*DT/EPSILON)
        ELSE
           KSI(1:IDIM4I,1:IDIM4J) = WORK(1:IDIM4I,1:IDIM4J,1)
           IF (SPONGE == 1)  SPONGE_MASK = EXP(-SPONGE_MASK*DT/EPSILON) 
           IF (WALLS /= 0)  WALL_MASK = EXP(-WALL_MASK*DT/EPSILON)
        END IF
     END IF

     
!     $$ II. 9. CALCULATING DIVERGENCE
!________________________________________________________________________________________________

#ifdef MPEG
     IF ((ISDIV==1 .AND. MOD(N-1,NSDIV)==0).OR.(IMPGDIV==1 .AND. MOD(N-1,NMPGDIV)==0)) THEN  
#else
     IF (ISDIV==1 .AND. MOD(N-1,NSDIV)==0) THEN 
#endif
        CALL ODIV (PU, PV, PD) 
     ENDIF

!     $$ II. 10. CALCULATING STREAMFUNCTION 
!________________________________________________________________________________________________

#ifdef MPEG
     IF ((ISSF==1 .AND. MOD(N-1,NSSF)==0).OR.(IMPGSF==1 .AND. MOD(N-1,NMPGSF)==0)) THEN
#else
     IF (ISSF==1 .AND. MOD(N-1,NSSF)==0) THEN
#endif   
        CALL OLAPINV(PO,PF)
     END IF


!     $$ II. 11. SAVING FIELDS
!___________________________________________________________________
!
!          $$ II.11.1.   RETURNING TO PHYSICAL SPACE
!          ------------------------------------

!CDIR IEXPAND
     CALL FFTCR (PU) 
!CDIR IEXPAND
     CALL FFTCR (PV) 
!CDIR IEXPAND
     CALL FFTCR (PP) 
!CDIR IEXPAND
     CALL FFTCR (PO) 
!CDIR IEXPAND
     CALL FFTCR (PF) 
     IF ( INCOMP == 0 ) THEN
!CDIR IEXPAND
        CALL FFTCR (PD) 
     ENDIF

!          $$ II.11.2.   SAVING SELECTED FIELDS
!          ------------------------------------
     CALL SAVE_PHYSICAL_FIELDS( N )

!     $$ II. 12. CALCULATING DIAGNOSTICS
!___________________________________________________________________
!
     CALL CALCULATE_DIAGNOSTICS( N )


!     $$ II. 13. PRODUCTS IN PHYSICAL SPACE
!___________________________________________________________________
!
!     P*U, P*V, O*U, O*V ET P+1/2(U2+V2)

!$OMP PARALLEL DO
     DO I=1,IDIM4J
        WORK(1:IDIM4I,I,1) = PP(1:IDIM4I,I)
        PF(1:IDIM4I,I) = PO(1:IDIM4I,I) * PV(1:IDIM4I,I)
     ENDDO

     IF ((FLOW_CASE == 1) .AND. (IROT /= 0)) THEN
!$OMP PARALLEL DO
        DO I=1,IDIM4J
           PO(1:IDIM4I,I) = PO(1:IDIM4I,I) * PU(1:IDIM4I,I) - CORIOL*U0
        ENDDO
     ELSE
!$OMP PARALLEL DO
        DO I=1,IDIM4J        
           PO(1:IDIM4I,I) = PO(1:IDIM4I,I) * PU(1:IDIM4I,I)
        ENDDO
     END IF
!$OMP PARALLEL DO
        DO I=1,IDIM4J        
           PP(1:IDIM4I,I) = WORK(1:IDIM4I,I,1) + ((PU(1:IDIM4I,I))**2 + (PV(1:IDIM4I,I))**2)*0.5
        ENDDO

     IF ( INCOMP == 0 ) THEN
!$OMP PARALLEL DO
        DO I=1,IDIM4J        
           PU(1:IDIM4I,I) = (WORK(1:IDIM4I,I,1) + MEAN_PRESS)*PU(1:IDIM4I,I)
           PV(1:IDIM4I,I) = (WORK(1:IDIM4I,I,1) + MEAN_PRESS)*PV(1:IDIM4I,I)
        ENDDO
     ENDIF

!     $$ II. 14. RETURNING TO SPECTRAL SPACE
!__________________________________________
!

     IF ( INCOMP == 0 ) THEN
!CDIR IEXPAND
        CALL FFTRC (PU) 
!CDIR IEXPAND
        CALL FFTRC (PV) 
     ENDIF
!CDIR IEXPAND
     CALL FFTRC (PP) 
!CDIR IEXPAND
     CALL FFTRC (PO) 
!CDIR IEXPAND
     CALL FFTRC (PF)


!     $$ II. 15. CALCULATING SPACE DERIVATIVES
!____________________________________________
!
!     DU/DT in the array PF
!     DU/DT=F-DP/DX
!     in spectral representation (DU/DT) = F-I*K*P
!     DV/DT in the array PO
!     DV/DT=-O-DP/DY
!     in spectral representation (DV/DT) = -O-I*H*P
!     DP/DT in the array PP (the incompressible case only)
!     DP/DT=-DU/DX-DV/DY
!     in spectral representation (DP/DT) = -I*K*U-I*H*V

!CDIR NODEP
!$OMP PARALLEL DO PRIVATE(K)
     DO H = 2, IDIM4J, 2 
!CDIR NODEP
        DO K = 2, IDIM4I, 2 
           PF(K-1,H-1) = PF(K-1,H-1) + (K - 2)*0.5*PP(K,H-1)               
           PF(K,H-1) = PF(K,H-1) + (2 - K)*0.5*PP(K-1,H-1) 
        END DO
!CDIR NODEP         
        DO K = 2, IDIM4I, 2
           PF(K-1,H) = PF(K-1,H) + (K - 2)*0.5*PP(K,H)
           PF(K,H) = PF(K,H) + (2 - K)*0.5*PP(K-1,H)
        END DO
     END DO

!$OMP PARALLEL DO PRIVATE(K)
!CDIR NODEP
     DO H = 2, IDIM4J, 2
!CDIR NODEP
        DO K = 2, IDIM4I, 2
           PO(K-1,H-1) = (-PO(K-1,H-1)) + ASPECT*(H - 2)*0.5*PP(K-1,H) 
           PO(K,H-1) = (-PO(K,H-1)) + ASPECT*(H - 2)*0.5*PP(K,H) 
        END DO
!CDIR NODEP            
        DO K = 2, IDIM4I, 2
           PO(K-1,H) = (-PO(K-1,H)) + ASPECT*(2 - H)*0.5*PP(K-1,H-1) 
           PO(K,H) = (-PO(K,H)) + ASPECT*(2 - H)*0.5*PP(K,H-1) 
        END DO
     END DO

! In the incompressible case only
     IF ( INCOMP == 0 ) THEN
!CDIR NODEP
!$OMP PARALLEL DO PRIVATE(K)
        DO H = 2, IDIM4J, 2 
!CDIR NODEP
           DO K = 2, IDIM4I, 2         
              PP(K-1,H-1) = ((K - 2)*PU(K,H-1)+ASPECT*(H-2)*PV(K-1,H))*0.5  
              PP(K,H-1) = ((2 - K)*PU(K-1,H-1)+ASPECT*(H-2)*PV(K,H))*0.5
           END DO
!CDIR NODEP            
           DO K = 2, IDIM4I, 2
              PP(K-1,H) = ((K - 2)*PU(K,H)+ASPECT*(2-H)*PV(K-1,H-1))*0.5  
              PP(K,H) = ((2 - K)*PU(K-1,H)+ASPECT*(2-H)*PV(K,H-1))*0.5 
           END DO
        END DO
     END IF

!     $$ II. 16.  TEMPORAL INTEGRATION
!___________________________________________
!
!           $$ II.16.1.   EXACT INTEGRATION OF THE PENALTY TERM
!           ---------------------------------------------------

     IF (PENALIZATION /= 0) CALL TPENA(FLOW_CASE)            

!           $$ II.16.2.   LEAPFROG / ADAMS-BASHFORTH
!           --------------------------------------------------

     IF ( INCOMP == 0 .OR. ( INCOMP == 1 .AND. SYNCHRO == 0 ) ) THEN
        CALL TLEAPFRO
     ELSE
        IF ( SYNCHRO == 1 ) THEN
! When resynchronizing, Adams-Bashforth reduces to explicit Euler;
!$OMP PARALLEL DO
           DO I=1,IDIM4J
              PUAA(1:IDIM4I,I) = PF(1:IDIM4I,I)
              PVAA(1:IDIM4I,I) = PO(1:IDIM4I,I)
           ENDDO
        ENDIF
        CALL TADAMS_BASHFORTH
     ENDIF

!           $$ II.16.3.   INCOMPRESSIBILITY VIA PRESSURE CORRECTION
!           --------------------------------------------------

     IF ( INCOMP > 0 ) CALL PRESSURE_CORRECTION( N )   

!     $$ II. 17.  RESYNCHRONIZATION OF LEAPFROG TIME STEPPING
!____________________________________________________________

     IF ( ISYNC == 1 .AND. SEPARLF >= ECARLF ) THEN 
        IF ( INCOMP == 0 ) THEN
! Leapfrog - compressible;
           CALL MODPRO (PUA, PVA, PPA, PU, PV, PP, DT, MEAN_PRESS, CORIOL) 
        ELSE IF ( INCOMP == 1 ) THEN
! Leapfrom - incompressible;
! synchronization without averaging;
!!$        PU(1:IDIM2I,1:IDIM4J) = 0.5 * ( PU(1:IDIM2I,1:IDIM4J) + PUA(1:IDIM2I,1:IDIM4J) )
!!$        PV(1:IDIM2I,1:IDIM4J) = 0.5 * ( PV(1:IDIM2I,1:IDIM4J) + PVA(1:IDIM2I,1:IDIM4J) )
!!$        PP(1:IDIM2I,1:IDIM4J) = 0.5 * ( PP(1:IDIM2I,1:IDIM4J) + PPA(1:IDIM2I,1:IDIM4J) )
!$OMP PARALLEL DO
           DO I=1,IDIM4J
              PUA(1:IDIM2I,I) = PU(1:IDIM2I,I)
              PVA(1:IDIM2I,I) = PV(1:IDIM2I,I)
              PPA(1:IDIM2I,I) = PP(1:IDIM2I,I)
           ENDDO
           IF ( VERBOSE > 0 ) WRITE(OUTPUT1, 4001) N, SEPARLF
! Will need to make a single AB step to synchronize leapfrog
           SYNCHRO = 1
        ENDIF
        SEPARLF = 0.0
     ELSE
! No need to resynchronize;
        SYNCHRO = 0
     ENDIF

!     $$ II. 18. SAVING SPECTRAL FIELDS ON DISC
!___________________________________________________________
!
     IF ( ISAVE == 1 .AND. (MOD(N,NSAVE) == 0 .OR. N == NM) ) CALL SAVE_SPECTRAL_FIELDS( N )


!     $$ II. 19.  END OF TIME STEPPING LOOP 
!___________________________________________

  END DO

!-----------------------------------------------
!
! III. WRAP-UP
!
!-----------------------------------------------
!
!     $$ III. 1. DETERMINING EXECUTION TIME
!_________________________________________________
 
#ifdef CRAY
TOP_FIN = SECOND()
#else
#ifdef SX
CALL CLOCK(TOP_FIN)
#else
#ifdef LINUX
CALL CPU_Time(TOP_FIN)
#else 
TOP_FIN = TOP_INI
#endif
#endif
#endif

TTIME = (TOP_FIN - TOP_INI)/NM 
WRITE(OUTPUT1,4002) TTIME

!     $$ III. 2  FLUSHING DATA
!___________________________________________________________________________________

IF ( PENALIZATION /= 0 ) THEN
   CLOSE (NFCX)
   CLOSE (NFCY)
ENDIF

!     $$ III. 3. CLOSING OUTPUT FILES
!_____________________________________________________________
! 
CLOSE(NFDIAG)
   
STOP  

!     $$ III. 4. OUTPUT FORMAT DEFINITIONS
!____________________________________

 4001 FORMAT('  Resynchronizing Leapfrog at N=',I7,'  separation=', F8.4)
 4002 FORMAT('  Time [sec] per time-step ', F10.6)


END PROGRAM TURB2D 


!-----------------------------------------------
!
! SUBROUTINES
!
!-----------------------------------------------

!_______________________________________________________________________
!
!                 Naming convention for subroutines
!                 ________________________________
!                    I   Initialization
!                    D   Diagnostics
!                    T   Temporal integration
!                    M   Normal modes
!                    F   Fourier Transforms
!                    O   Operators
!_______________________________________________________________________


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  INIT_PARAMETERS (Bartek Protas, 16/12/2003)
!
! PARAMETERS:
!  - NONE
!
! DESCRIPTION:
!  - reads parameters from the input file
!  - opens streams for output
!  - writes out the parameters as a check
!______________________________________________________________________

SUBROUTINE INIT_PARAMETERS
 
  USE parameters
  USE database
  USE diagnostics
  IMPLICIT NONE

  WRITE(OUTPUT1,*)'  Reading in the parameters'

! reading parameters from file
! instruction for IDRIS
!  OPEN(UNIT=5, STATUS="OLD", FILE="parameter",ACTION="READ",FORM="FORMATTED")
  READ (NFPARAM, 5000) VERBOSE	
  READ (NFPARAM, 5000) IINIT 	
  READ (NFPARAM, 5000) INCOMP 
  READ (NFPARAM, 5000) IDALIAS 
  READ (NFPARAM, 5000) ISYNC 
  READ (NFPARAM, 5000) IHYPER
  READ (NFPARAM, 5000) FLOW_CASE
  READ (NFPARAM, 5000) IVISC  
  READ (NFPARAM, 5000) IROT 
  READ (NFPARAM, 5000) IFORCE 
  READ (NFPARAM, 5000) ISAVE 
  READ (NFPARAM, 5000) ISVORT
  READ (NFPARAM, 5000) ISVPOT
  READ (NFPARAM, 5000) ISPRESS
  READ (NFPARAM, 5000) ISDIV
  READ (NFPARAM, 5000) ISSF
  READ (NFPARAM, 5000) ISU
  READ (NFPARAM, 5000) ISV
  READ (NFPARAM, 5000) ISMODV
  READ (NFPARAM, 5000) ISBERN
  READ (NFPARAM, 5000) ISCUTP
  READ (NFPARAM, 5000) ISCUTV
  READ (NFPARAM, 5000) ISCUTU
  READ (NFPARAM, 5000) NUMCUT
  READ (NFPARAM, 5000) NM 
  READ (NFPARAM, 5000) NINV 
  READ (NFPARAM, 5000) NSAVE 
  READ (NFPARAM, 5000) NSVORT
  READ (NFPARAM, 5000) NSVPOT
  READ (NFPARAM, 5000) NSPRESS
  READ (NFPARAM, 5000) NSDIV 
  READ (NFPARAM, 5000) NSSF
  READ (NFPARAM, 5000) NSU
  READ (NFPARAM, 5000) NSV 
  READ (NFPARAM, 5000) NSMODV 
  READ (NFPARAM, 5000) NSBERN
  READ (NFPARAM, 5000) NSCUT
  READ (NFPARAM, 5001) FLUCT 
  READ (NFPARAM, 5001) ECARLF 
  READ (NFPARAM, 5001) CFL 
  READ (NFPARAM, 5000) KEXCIT 
  READ (NFPARAM, 5001) BURGER
  READ (NFPARAM, 5001) ROSSBY
  READ (NFPARAM, 5001) REYNOLDS
  READ (NFPARAM, 5000) PENALIZATION
  READ (NFPARAM, 5000) SPONGE
  READ (NFPARAM, 5000) WALLS
  READ (NFPARAM, 5000) FORC_VMEAN
  READ (NFPARAM, 5001) COEF_DIAM_LO
  READ (NFPARAM, '(1PE10.3)') NU_COEFNU
   
#ifdef MPEG
  READ (NFPARAM, 5000) IMPGVORT
  READ (NFPARAM, 5000) IMPGVPOT
  READ (NFPARAM, 5000) IMPGPRESS
  READ (NFPARAM, 5000) IMPGDIV 
  READ (NFPARAM, 5000) IMPGSF
  READ (NFPARAM, 5000) IMPGU
  READ (NFPARAM, 5000) IMPGV
  READ (NFPARAM, 5000) IMPGMODV
  READ (NFPARAM, 5000) IMPGBERN
  
  READ (NFPARAM, 5000) NMPGVORT
  READ (NFPARAM, 5000) NMPGVPOT
  READ (NFPARAM, 5000) NMPGPRESS
  READ (NFPARAM, 5000) NMPGDIV 
  READ (NFPARAM, 5000) NMPGSF
  READ (NFPARAM, 5000) NMPGU
  READ (NFPARAM, 5000) NMPGV
  READ (NFPARAM, 5000) NMPGMODV
  READ (NFPARAM, 5000) NMPGBERN
#endif   

! Adjusting parameters if inconsistencies are found (incompressible case);
  IF ( VERBOSE > 1 )  WRITE(OUTPUT1,*)'  Adjusting parameters, if needed'
  IF ( INCOMP > 0 ) THEN
     IF ( IROT /= 0 ) THEN
        WRITE(STDERR,*) '    Turning off background rotation'
        IROT = 0
     ENDIF
     IF ( ISVPOT /= 0 ) THEN
        WRITE(STDERR,*) '    Turning off saving potential vorticity'
        ISVPOT = 0
        IMPGVPOT = 0
     ENDIF
     IF ( ISDIV /= 0 ) THEN
        WRITE(STDERR,*) '    Turning off saving divergence'
        ISDIV = 0
        IMPGDIV = 0
     ENDIF
     IF ( FORC_VMEAN /= 0 ) THEN
        WRITE(STDERR,*) '    Turning off zonal velocity forcing'
        FORC_VMEAN = 0
     ENDIF
     IF ( IFORCE == 1 ) THEN
        WRITE(STDERR,*) '    Turning off pressure forcing'
        IFORCE = 0
     ENDIF
     IF ( ISYNC == 1 .AND. INCOMP == 2 ) THEN
        WRITE(STDERR,*) '    Turning off resynchronization when using Adams-Bashforth'
        ISYNC = 0
     ENDIF

! Turning-off movies
     IF ( VERBOSE == 0 ) THEN
        IF ( IMPGVORT == 1 )  IMPGVORT  = 0
        IF ( IMPGVPOT == 1 )  IMPGVPOT  = 0
        IF ( IMPGPRESS == 1 ) IMPGPRESS = 0
        IF ( IMPGDIV == 1 )   IMPGDIV   = 0
        IF ( IMPGSF == 1 )    IMPGSF    = 0
        IF ( IMPGU == 1 )     IMPGU     = 0
        IF ( IMPGV == 1 )     IMPGV     = 0
        IF ( IMPGMODV == 1 )  IMPGMODV  = 0
        IF ( IMPGBERN == 1 )  IMPGBERN  = 0
        WRITE(STDERR,*) '    Turning off movies'
     END IF
  ENDIF

  IF ((FLOW_CASE == 3) .AND. (SPONGE == 1)) THEN
     WRITE(STDERR,*) 'No sponge if the obstacle is translated'
     SPONGE = 0
  ENDIF

  IF ((PENALIZATION == 0) .AND. ((SPONGE == 1) .OR. (WALLS == 1))) THEN
     WRITE(STDERR,*) 'To have sponge or walls, penalization must be different to zero'
     PENALIZATION = 6
  ENDIF

  IF ((INCOMP==0).AND.(IROT == 0)) THEN
     WRITE(STDERR,*) 'Turning on background rotation'
     IROT=1
  ENDIF

  IF ( VERBOSE > 1 ) WRITE(OUTPUT1,*) '  Determine flow case'
 
  SELECT CASE ( FLOW_CASE )
  CASE(0)
     IF ( INCOMP == 0 ) THEN
! This is the compressible (SW), isotropic homogeneous case
        TITLE = 'BAV0R0'
     ELSE
! This is the incompressible (NS), isotropic homogeneous case
        TITLE = 'IRV0R0'
     ENDIF
  CASE(1)      
! The obstacle is translated but the referential is the obstacle's one
     TITLE = 'OBSFIX'
  CASE(2)
! The obstacle is fixed and an upstream current at geostrophic balance hit it
     TITLE = 'OFEQGE'
  CASE(3)
! The obstacle is translated and the referential is the tank's one
     TITLE = 'OBSMOB'
  CASE(4)
     TITLE = 'STABIL'
  CASE DEFAULT
     TITLE = '00????'
  END SELECT

  IF ( ( FLOW_CASE <= 3 ) .AND. ( FLOW_CASE >= 0 ) ) THEN
     WRITE (OUTPUT1, 6039) TITLE, FLOW_CASE
  ELSE
     WRITE (OUTPUT1,'("The read parameters don''t correspond to any predifined title")')
  ENDIF

  IF ( VERBOSE > 1 )  WRITE(OUTPUT1,*)'  Opening files'

  FNAME = TRIM(TITLE)//'.dat'
  OPEN(STATUS="REPLACE", UNIT=NFDIAG, FILE=FNAME, FORM="FORMATTED")

!!/////////////////////////////////////////////////////////////////////////
!!///// Update made by Bartek Protas on the 17th May  2004 ////////////////
!!$  IF (ISVORT==1)  OPEN(STATUS="REPLACE", UNIT=NFVORT, FILE="vorticity",  ACCESS="DIRECT", FORM="UNFORMATTED")
!!$  IF (ISVPOT==1)  OPEN(STATUS="REPLACE", UNIT=NFVPOT, FILE="potential_vorticity", ACCESS="DIRECT", FORM="UNFORMATTED")
!!$  IF (ISPRESS==1) OPEN(STATUS="REPLACE", UNIT=NFPRESS, FILE="pressure", ACCESS="DIRECT", FORM="UNFORMATTED")
!!$  IF (ISDIV==1)   OPEN(STATUS="REPLACE", UNIT=NFDIV, FILE="divergence", ACCESS="DIRECT", FORM="UNFORMATTED")
!!$  IF (ISSF==1)    OPEN(STATUS="REPLACE", UNIT=NFSF, FILE="streamfunction", ACCESS="DIRECT", FORM="UNFORMATTED")
!!$  IF (ISU==1)     OPEN(STATUS="REPLACE", UNIT=NFU, FILE="v_x", ACCESS="DIRECT", FORM="UNFORMATTED")
!!$  IF (ISV==1)     OPEN(STATUS="REPLACE", UNIT=NFV, FILE="v_y", ACCESS="DIRECT", FORM="UNFORMATTED")
!!$  IF (ISMODV==1)  OPEN(STATUS="REPLACE", UNIT=NFMODV,  FILE="modV", ACCESS="DIRECT", FORM="UNFORMATTED")     
!!$  IF (ISBERN==1)  OPEN(STATUS="REPLACE", UNIT=NFBERN, FILE="bernouilli", ACCESS="DIRECT", FORM="UNFORMATTED")  
!!$  IF (ISCUTP==1)  OPEN(STATUS="REPLACE", UNIT=NFCUTP, FILE="slice_pressure", ACCESS="DIRECT", FORM="UNFORMATTED")
!!$  IF (ISCUTV==1)  OPEN(STATUS="REPLACE", UNIT=NFCUTV, FILE="slice_vy", ACCESS="DIRECT", FORM="UNFORMATTED")
!!$  IF (ISCUTU==1)  OPEN(STATUS="REPLACE", UNIT=NFCUTU, FILE="slice_vx", ACCESS="DIRECT", FORM="UNFORMATTED")
!!$  IF (PENALIZATION/=0) THEN
!!$     OPEN(STATUS="REPLACE", UNIT=NFCX, FILE="lift", ACCESS="DIRECT", FORM="UNFORMATTED")
!!$     OPEN(STATUS="REPLACE", UNIT=NFCY, FILE="drag", ACCESS="DIRECT", FORM="UNFORMATTED")  
!!$  ENDIF

  IF (ISVORT==1)  OPEN(STATUS="REPLACE", UNIT=NFVORT, FILE="vorticity", FORM="BINARY")
  IF (ISVPOT==1)  OPEN(STATUS="REPLACE", UNIT=NFVPOT, FILE="potential_vorticity", FORM="BINARY")
  IF (ISPRESS==1) OPEN(STATUS="REPLACE", UNIT=NFPRESS, FILE="pressure", FORM="BINARY")
  IF (ISDIV==1)   OPEN(STATUS="REPLACE", UNIT=NFDIV, FILE="divergence", FORM="BINARY")
  IF (ISSF==1)    OPEN(STATUS="REPLACE", UNIT=NFSF, FILE="streamfunction", FORM="BINARY")
  IF (ISU==1)     OPEN(STATUS="REPLACE", UNIT=NFU, FILE="v_x", FORM="BINARY")
  IF (ISV==1)     OPEN(STATUS="REPLACE", UNIT=NFV, FILE="v_y", FORM="BINARY")
  IF (ISMODV==1)  OPEN(STATUS="REPLACE", UNIT=NFMODV,  FILE="modV", FORM="BINARY")     
  IF (ISBERN==1)  OPEN(STATUS="REPLACE", UNIT=NFBERN, FILE="bernouilli", FORM="BINARY")  
  IF (ISCUTP==1)  OPEN(STATUS="REPLACE", UNIT=NFCUTP, FILE="slice_pressure", FORM="BINARY")
  IF (ISCUTV==1)  OPEN(STATUS="REPLACE", UNIT=NFCUTV, FILE="slice_vy", FORM="BINARY")
  IF (ISCUTU==1)  OPEN(STATUS="REPLACE", UNIT=NFCUTU, FILE="slice_vx", FORM="BINARY")
  IF (PENALIZATION/=0) THEN
     OPEN(STATUS="REPLACE", UNIT=NFCX, FILE="lift", FORM="BINARY")
     OPEN(STATUS="REPLACE", UNIT=NFCY, FILE="drag", FORM="BINARY")  
  ENDIF
!!/////////////////////////////////////////////////////////////////////////

#ifdef MPEG
  IF (IMPGVORT==1)  CALL OpenMovieOut(MPGVORT,"vort",IDIM3I,IDIM3J)
  IF (IMPGVPOT==1)  CALL OpenMovieOut(MPGVPOT,"vpot",IDIM3I,IDIM3J)
  IF (IMPGPRESS==1) CALL OpenMovieOut(MPGPRESS,"pres",IDIM3I,IDIM3J)
  IF (IMPGDIV==1)   CALL OpenMovieOut(MPGDIV,"dive",IDIM3I,IDIM3J)
  IF (IMPGSF==1)    CALL OpenMovieOut(MPGSF,"cour",IDIM3I,IDIM3J)
  IF (IMPGU==1)     CALL OpenMovieOut(MPGU,"Uvel",IDIM3I,IDIM3J)
  IF (IMPGV==1)     CALL OpenMovieOut(MPGV,"Vvel",IDIM3I,IDIM3J)
  IF (IMPGMODV==1)  CALL OpenMovieOut(MPGMODV,"modv",IDIM3I,IDIM3J)
  IF (IMPGBERN==1)  CALL OpenMovieOut(MPGBERN,"bern",IDIM3I,IDIM3J)
#endif
  
! Writing out the parameters as a check
  IF ( VERBOSE > 2 ) THEN
     WRITE (OUTPUT1, 6050) VERBOSE, IINIT, INCOMP, IDALIAS, ISYNC, IHYPER, FLOW_CASE,&
          IVISC,IROT, IFORCE, ISAVE, ISVORT, ISVPOT, ISPRESS, ISDIV, &
          ISSF, ISU, ISV, ISMODV, ISBERN, ISCUTP, ISCUTV, ISCUTU,NUMCUT     
     WRITE (OUTPUT1, 6051) NM, NINV, NSAVE, NSVORT, &
          NSVPOT, NSPRESS, NSDIV, NSSF, NSU, NSV, NSMODV, NSBERN, NSCUT, &
          FLUCT, ECARLF, CFL, KEXCIT, BURGER, ROSSBY, REYNOLDS, &
          PENALIZATION, SPONGE, WALLS, FORC_VMEAN, COEF_DIAM_LO
#ifdef MPEG
      WRITE (OUTPUT1, '(I5,3X,"IMPGVORT")') IMPGVORT
      WRITE (OUTPUT1, '(I5,3X,"IMPGVPOT")') IMPGVPOT
      WRITE (OUTPUT1, '(I5,3X,"IMPGPRESS")') IMPGPRESS
      WRITE (OUTPUT1, '(I5,3X,"IMPGDIV")') IMPGDIV 
      WRITE (OUTPUT1, '(I5,3X,"IMPGSF")') IMPGSF
      WRITE (OUTPUT1, '(I5,3X,"IMPGU")') IMPGU
      WRITE (OUTPUT1, '(I5,3X,"IMPGV")') IMPGV
      WRITE (OUTPUT1, '(I5,3X,"IMPGMODV")') IMPGMODV
      WRITE (OUTPUT1, '(I5,3X,"IMPGBERN")') IMPGBERN
      WRITE (OUTPUT1, '(I5,3X,"NMPGVORT")') NMPGVORT
      WRITE (OUTPUT1, '(I5,3X,"NMPGVPOT")') NMPGVPOT
      WRITE (OUTPUT1, '(I5,3X,"NMPGPRESS")') NMPGPRESS
      WRITE (OUTPUT1, '(I5,3X,"NMPGDIV")') NMPGDIV 
      WRITE (OUTPUT1, '(I5,3X,"NMPGSF")') NMPGSF
      WRITE (OUTPUT1, '(I5,3X,"NMPGU")') NMPGU
      WRITE (OUTPUT1, '(I5,3X,"NMPGV")') NMPGV
      WRITE (OUTPUT1, '(I5,3X,"NMPGMODV")') NMPGMODV
      WRITE (OUTPUT1, '(I5,3X,"NMPGBERN")') NMPGBERN
#endif 
   ENDIF


 6039 FORMAT(////66('_'),/,20X,'Flow case:',2X,A6,'(',I2,')',/,66('_')) 
 5000 FORMAT(I7) 
 5001 FORMAT(F7.2)
 6050 FORMAT(///,20X,'INPUT PARAMETERS',2/,20X,I7,5X,'VERBOSE ',/,20X,I7,5X,'IINIT ',   &
           /,20X,I7,5X,'INCOMP',/,20X,I7,5X,'IDALIAS',/,20X,I7,5X,'ISYNC ',/,20X,I7,5X,  &
           'IHYPER',/,20X,I7,5X,'FLOW_CASE',/,20X,I7,5X,'IVISC ',/,20X,I7,5X,'IROT  ',&
           /,20X,I7,5X,'IFORCE',/,20X,I7,5X,'ISAVE',/,20X,I7,5X,'ISVORT',/,20X,I7,5X,&
           'ISVPOT',/,20X,I7,5X,'ISPRESS',/,20X,I7,5X,'ISDIV',/,20X,I7,5X,'ISSF',/,20X,I7,5X,&
           'ISU',/,20X,I7,5X,'ISV',/,20X,I7,5X,'ISMODV', /,20X,I7,5X,'ISBERN',/,20X,I7,5X,'ISCUTP' &
           ,/,20X,I7,5X,'ISCUTV',/,20X,I7,5X,'ISCUTU',/,20X,I7,5X,'NUMCUT')
 6051 FORMAT(20X,I7,5X,'NM    ',/,20X,I7,5X,'NINV  ',/,20X,I7,5X,&
         'NSAVE',/,20X,I7,5X,'NSVORT',/,20X,I7,5X,'NSVPOT',/,20X,I7,5X,&
         'NSPRESS',/,20X,I7,5X,'NSDIV',/,20X,I7,5X,'NSSF',/,20X,I7,5X,&
         'NSU',/,20X,I7,5X,'NSV',/,20X,I7,5X,'NSMODV',/,20X,I7,5X, &
	 'NSBERN',/,20X,I7,5X,'NSCUT',/,20X,F7.2,5X,'FLUCT ',/,20X,F7.2,5X,&
	 'ECARLF',/,20X,F7.2,5X,'CFL   ',/,20X,I7,5X,'KEXCIT',/,20X,F7.2,5X,&
	 'BURGER ',/,20X,F7.2,5X,'ROSSBY ',/,20X,F7.2,5X,'REYNOLDS ',3/,20X,I7,5X,&
	 'PENALIZATION',/,20X,I7,5X,'SPONGE',/,20X,I7,5X,'WALLS',3/,20X,I7,5X,'FORCE_VMOY',&
         /,20X,F7.2,5X,'COEF_DIAM_LO',3/)


END SUBROUTINE INIT_PARAMETERS


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  INIT_DATA (Bartek Protas, 16/12/2003)
!
! PARAMETERS:
!  - NONE
!
! DESCRIPTION:
!  - initializes variables
!  - allocates dynamic tables
!  - computes and assigns all parameters
!______________________________________________________________________

SUBROUTINE INIT_DATA
 
  USE parameters
  USE database
  USE diagnostics
  IMPLICIT NONE


!     1. INITIALIZING DATA
!__________________________


  IF ( VERBOSE > 1 )  WRITE(OUTPUT1,*)'  Initializing data'

  NINIT     = 0
  NTIME     = 0 
  NSTEP     = 0  
  TIME0     = 0.0  
  TIMEA     = 0.0  
  TIME      = 0.0

! Initialization of the arrays;

  PUAA(1:IDIM4I,1:IDIM4J) = 0.
  PVAA(1:IDIM4I,1:IDIM4J) = 0.
  PPAA(1:IDIM4I,1:IDIM4J) = 0.
  PCAA(1:IDIM4I,1:IDIM4J) = 0.
  PUA(1:IDIM4I,1:IDIM4J)  = 0.
  PVA(1:IDIM4I,1:IDIM4J)  = 0.
  PPA(1:IDIM4I,1:IDIM4J)  = 0.
  PCA(1:IDIM4I,1:IDIM4J)  = 0.
  PU(1:IDIM4I,1:IDIM4J)   = 0.
  PV(1:IDIM4I,1:IDIM4J)   = 0.
  PP(1:IDIM4I,1:IDIM4J)   = 0.
  PO(1:IDIM4I,1:IDIM4J)   = 0.
  PF(1:IDIM4I,1:IDIM4J)   = 0.
  PD(1:IDIM4I,1:IDIM4J)   = 0.
  PC(1:IDIM4I,1:IDIM4J)   = 0.
  MASK(1:IDIM4I,1:IDIM4J) = 0.
  WORK(1:IDIM4I,1:IDIM4J,1:2) = 0.  
  DISSIP(1:IDIM4I,1:IDIM4J) = 1. 
  KSI(1:IDIM4I,1:IDIM4J) = 0.
  GEOP_FORCE(1:IDIM4I,1:IDIM4J) = 0.
  SPONGE_MASK(1:IDIM4I,1:IDIM4J) = 0.
  WALL_MASK(1:IDIM4I,1:IDIM4J) = 0.

! Various forms of energy;
  ELIN  = 0.0
  EPOT  = 0.0
  EKIN  = 0.0
  ETOT  = 0.0
  ZPOT  = 0.0  
  ZLIN  = 0.0
  ZTOT  = 0.0

! allocating dynamic tables;
  ALLOCATE (DRAG(NM,2))
  DRAG = 0.0

!  INITIALIZING PHYSICAL DATA

! Geometric specifications

  ASPECT=REAL(IDIM3I)/REAL(IDIM3J)
  ASPECT2=ASPECT*ASPECT

! Physical parameters

  PI =  2.*ASIN(1.)
  L0 = 2*PI
  P0 = 1.0
  IF (IDIM3I <= IDIM3J) THEN
     DIAMETER = COEF_DIAM_LO * L0
  ELSE
     DIAMETER = COEF_DIAM_LO * L0 / ASPECT
  END IF
  IF ((INCOMP == 0) .OR. (PENALIZATION /= 0)) THEN
     U0 = (P0**0.5)*ROSSBY/(BURGER**0.5)
  ELSE
     U0 = 1.0
  END IF
  IF (INCOMP == 1) P0 = 0.0

  FLUCTP = FLUCT 
  FLUCTV = FLUCT 

! Rotation parameters

! RADDEF deformation radius = R * SQRT(BU) = L0 / KDEF
! CORIOL Coriolis parameter = SQRT(P0)/RADDEF
  IF (IROT == 0) THEN 
     CORIOL = 1.E-10
     RADDEF = 1.E+10   
  ELSE 
     RADDEF = (DIAMETER/2)*SQRT(BURGER)
     CORIOL = SQRT(P0)/RADDEF 
  ENDIF     
  CORIOL2 = CORIOL*CORIOL 

! NUMERICAL PARAMETERS

! Viscosity

  SELECT CASE ( IVISC )
    CASE ( 0 )
! the inviscid case;
       NU = 0.0
    CASE ( 3 )
! computed based on the Reynolds number;
       NU  = U0 * DIAMETER / REYNOLDS
    CASE DEFAULT
       NU = NU_COEFNU
  END SELECT

! Dissipation exponent

  IF (IHYPER==1) THEN
     DISSIPATION_EXPONENT = 8 
  ELSE
     DISSIPATION_EXPONENT = 1
  END IF
  
! Resynchronization

  IF (ISYNC == 0) THEN
     ECARLF = 0.
  END IF

! Dealiasing

   IF (IDALIAS > 0) THEN 
     KCUTOFFI = IDIM3I/3
     KCUTOFFJ = IDIM3J/3
  ELSE 
     KCUTOFFI = IDIM3I/2
     KCUTOFFJ = IDIM3J/2  
  ENDIF
  KCUTOFF = MAX(KCUTOFFI,ASPECT*KCUTOFFJ)  
  CALL FMASK

! Total number of grid point
  NMPOINT = IDIM3I*IDIM3J 

! The largest wavenumber
  KMAX = KCUTOFF

! Space grid
  DX = L0 / (2*KMAX)

  IF ( VERBOSE > 2 )  WRITE(OUTPUT1,8010) FLUCTP, FLUCTV,  RADDEF, CORIOL, ASPECT, &
                                          NU, KCUTOFF, KMAX, DX, U0

8010 FORMAT(  /,6X,'FLUCTP   =',1PE14.6,/, &
                6X,'FLUCTV   =',1PE14.6,/, &
                6X,'RADDEF   =',1PE14.6,/, &
                6X,'CORIOL   =',1PE14.6,/, & 
                6X,'ASPECT   =',1PE14.6,/, &
                6X,'NU       =',1PE14.6,/, &
                6X,'KCUTOFF  =',1PE14.6,/, & 
                6X,'KMAX     =',1PE14.6,/, &
                6X,'DX       =',1PE14.6,/, &
                6X,'U0       =',1PE14.6,/// )

END SUBROUTINE INIT_DATA



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  INIT_FFT (Bartek Protas, 17/12/2003)
!
! PARAMETERS:
!  - NONE
!
! DESCRIPTION:
!  - initializes the FFTw wrapper
!______________________________________________________________________

SUBROUTINE INIT_FFT
  
  USE parameters, ONLY: VERBOSE
  USE database
  USE diagnostics, ONLY: OUTPUT1
  IMPLICIT NONE

  IF ( VERBOSE > 1 )  WRITE(OUTPUT1,*)'  Initializing FFT'
  
  JUMP1 = IDIM3I + 2 
  JUMP2 = 1
  INC1  = 1
  INC2  = JUMP1   
  
#if TURB2D_FFT_USED == SCILIB_CRAY

  CALL FFTFAX (IDIM3I, IFAXI, TRIGSI)
  CALL FFTFAX (IDIM3J, IFAXJ, TRIGSJ)

#elif TURB2D_FFT_USED == FFTW 

  CALL FFTWFAX (IDIM3I, IFAXI, TRIGSI)
  CALL FFTWFAX (IDIM3J, IFAXJ, TRIGSJ)

#elif TURB2D_FFT_USED == ASL_NEC

#endif

END SUBROUTINE INIT_FFT


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  INIT_FLOW (Bartek Protas, 17/12/2003)
!
! PARAMETERS:
!  - NONE
!
! DESCRIPTION:
!  - creates an initial flow field from scratch or from an input file
!  - determines the time step based on the CFL number
!______________________________________________________________________

SUBROUTINE INIT_FLOW
  
  USE database
  USE parameters
  USE diagnostics, ONLY: OUTPUT1, NFV
  IMPLICIT NONE

  INTEGER :: I, J
  
  IF (IINIT == 1) THEN
     IF ( VERBOSE > 1 )  WRITE(OUTPUT1,*)'  Creating the Initial Condition'     
! The flow field created from scratch, in Fourier space;
     SELECT CASE (FLOW_CASE)
     CASE(0) 
        IF ( INCOMP == 0 ) THEN
           CALL INBAL                        ! Isotropic homogeneous SW case
           PP(1,1) = PP(1,1) + P0            ! Initialization with ..
        ELSE
           CALL INROTA(FLUCT)                ! Isotropic homogeneous incompressible case
                                             ! Initialization with a random field
        ENDIF          
     CASE(1)                              ! Upstream hitting the island with a velocity U0
        CALL INUPSTREAM(U0,0.*U0)
        PP(1:IDIM4I,1:IDIM4J) = P0
        MEAN_PRESS = P0
     CASE(2)                              ! Upstream at geostrophic balance hitting the island with a velocity U0
        CALL IPRESSION_GEO(U0)
        PP(1,1) = P0
        MEAN_PRESS = P0
        
     CASE(3)                              ! Initial flow at rest
        CALL INUPSTREAM(0.*U0,0.*U0)
        PP(1:IDIM4I,1:IDIM4J) = P0
        MEAN_PRESS = P0
     CASE(4)
        CALL INISTAB
     END SELECT

! This could maybe be moved to the respective functions above ....
     IF ( FLOW_CASE > 0 ) THEN
        WORK(1:IDIM4I,1:IDIM4J,1) = SQRT(PU(1:IDIM4I,1:IDIM4J)**2+PV(1:IDIM4I,1:IDIM4J)**2) 
        V0MAX = MAXVAL(WORK(1:IDIM4I,1:IDIM4J,1))
     END IF

! Determining the time step
     IF ( INCOMP == 0 ) THEN
        DTC  = 1.0  / SQRT(KMAX*KMAX*P0 + CORIOL2) 
        CSON = 1.0 /(KMAX*DTC) 
        DT   = CFL * ( DX / (PI*(CSON + V0MAX)) ) 
     ELSE
        DT   = CFL / ( PI * KMAX * V0MAX )  
     ENDIF

! for the wall-bounded case;
     IF (PENALIZATION /=0) THEN
     
        EPSILON = DT * 2
        WRITE (OUTPUT1,'("  EPSILON = ")')
        WRITE (OUTPUT1,'(1PE14.6)') EPSILON
     
        CALL INIPENA(PENALIZATION,NSTEP)    

        IF (FLOW_CASE <= 2) THEN        ! The obstacle is at rest  
           PV = PV *(1-KSI)
           PU = PU *(1-KSI)
           KSI = EXP(-KSI*DT/EPSILON)
           IF (SPONGE == 1) SPONGE_MASK = EXP(-SPONGE_MASK*DT/EPSILON) 
           IF (WALLS /= 0)  WALL_MASK = EXP(-WALL_MASK*DT/EPSILON)
    
        ELSE IF (FLOW_CASE == 3) THEN   ! The obstacle is translated at the velocity u0force      
           PV = PV * (1 - KSI)
           PU = PU * (1 - KSI)
           VELO_FORCE = -KSI*U0/EPSILON
           KSI = EXP(-KSI*DT/EPSILON)
           IF (WALLS /= 0) WALL_MASK = EXP(-WALL_MASK*DT/EPSILON)
        END IF        
        IF ( FLOW_CASE/=2 ) THEN        
           CALL FFTRC (PU)
           CALL FFTRC (PV)
           CALL FFTRC (PP)
        ENDIF
     ENDIF
  ELSE
     IF ( VERBOSE > 1 )  WRITE(OUTPUT1,*)'  Reading the Initial Condition from a file '
! reading the initial field from a file (IINIT=0,2);
     READ (1)  NSTEP, TIMEA, TIME
     READ (1) ((PUA(I,J), I=1,IDIM4I), J=1,IDIM4J)
     READ (1) ((PVA(I,J), I=1,IDIM4I), J=1,IDIM4J)
     READ (1) ((PPA(I,J), I=1,IDIM4I), J=1,IDIM4J)
     READ (1) ((PU(I,J),  I=1,IDIM4I), J=1,IDIM4J)
     READ (1) ((PV(I,J),  I=1,IDIM4I), J=1,IDIM4J)
     READ (1) ((PP(I,J),  I=1,IDIM4I), J=1,IDIM4J)
     IF (PENALIZATION /= 0) READ (1) ((KSI(I,J), I=1,IDIM4I), J=1,IDIM4J)
     IF (SPONGE /= 0) READ (1) ((SPONGE_MASK(I,J), I=1,IDIM4I), J=1,IDIM4J)
     IF (WALLS /= 0) READ (1) ((WALL_MASK(I,J), I=1,IDIM4I), J=1,IDIM4J)
! Determining the time step;
     NSTEPA = NSTEP - 1 
     DT     = TIME - TIMEA
     EPSILON = 2*DT
     TIME0  = TIME 
     NINIT  = NSTEP  
! calculating the initial vorticity;
     CALL OROT ( PU, PV, PO )

! Going to real space to calculate fluctuations;
     CALL FFTCR (PU) 
     CALL FFTCR (PV) 
     CALL FFTCR (PO) 
     IF ( INCOMP == 0 ) CALL FFTCR (PP)
     
     CALL DFLUCT
     
     CALL FFTRC (PU) 
     CALL FFTRC (PV) 
     CALL FFTRC (PO) 
     IF ( INCOMP == 0 ) CALL FFTRC (PP)          
  ENDIF

! Calculating the dissipation matrices
  IF ( INCOMP < 2 ) CALL TDISSIP	
  IF ( INCOMP > 0 ) CALL TDISSIP2

  IF ( INCOMP == 0 ) THEN
! In the compressible case, elimination of the parasitic leapfrog eigenmodes;
     CALL MODPRO(PUA,PVA,PPA,PU,PV,PP,DT,P0,CORIOL) 

! IG modes eliminated in the compressible case only;
     IF ( IINIT == 2 ) THEN
        CALL MPROGE(PUA,PVA,PPA,P0,CORIOL)
        CALL MPROGE(PU,PV,PP,P0,CORIOL)
     ENDIF
     IF ( VERBOSE > 2 ) WRITE (OUTPUT1, 8011) CSON, V0MAX, DT
  ELSE     
     IF ( VERBOSE > 2 ) WRITE (OUTPUT1, 8012) V0MAX, DT
  ENDIF
  
! Going to real space, so that diagnostics can be computed;
  CALL FFTCR (PU) 
  CALL FFTCR (PV) 
  CALL FFTCR (PO) 
  IF ( INCOMP == 0 ) CALL FFTCR (PP)
  
  CALL CALCULATE_DIAGNOSTICS( 0 )

  CALL FFTRC (PU) 
  CALL FFTRC (PV) 
  CALL FFTRC (PO) 
  IF ( INCOMP == 0 ) CALL FFTRC (PP)          

8011 FORMAT(  /,6X,'VSOUND   =',1PE14.6,/, &
                6X,'VOMAX    =',1PE14.6,/, &
                6X,'DT       =',1PE14.6,/)

8012 FORMAT(  /,6X,'VOMAX    =',1PE14.6,/, &
                6X,'DT       =',1PE14.6,/)
 
END SUBROUTINE INIT_FLOW


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  INUPSTREAM
!
! PARAMETERS:
!  - U0 the initial transverse velocity
!  - V0 the initial longitudinal velocity
!
! DESCRIPTION:
!  - creates an initial field with an initial constant velocity
!______________________________________________________________________

SUBROUTINE INUPSTREAM(Uo,Vo) 

  USE database
  USE parameters
  USE diagnostics, ONLY: OUTPUT1

  IMPLICIT NONE

   REAL(KIND=SIZE_OF_REAL), INTENT(IN) :: Uo, Vo

  IF ( VERBOSE > 2 ) WRITE (OUTPUT1, 6000) 

  PU(1:IDIM4I,1:IDIM4J) = Uo
  PV(1:IDIM4I,1:IDIM4J) = Vo

  RETURN  
6000 FORMAT(///,1X,'INITIAL FLOW AT CONSTANT VELOCITY',/,1X,&
         '_______________________________',/) 
END SUBROUTINE INUPSTREAM



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  IPRESSION_GEO
!
! PARAMETERS:
!  - U0 the initial longitudinal velocity
!
! DESCRIPTION:
!  - creates an initial field with an initial constant velocity
!______________________________________________________________________

SUBROUTINE IPRESSION_GEO(Uo) 

  USE database
  USE parameters
  USE diagnostics, ONLY: OUTPUT1

  IMPLICIT NONE

   REAL(KIND=SIZE_OF_REAL), INTENT(IN) :: Uo
   INTEGER :: K, H

  IF ( VERBOSE > 2 ) WRITE (OUTPUT1, 6000) 

  PV(1,1) = Uo

!CALCUL DU CHAMP DE GEOPOTENTIEL GEOSTROPHIQUE ASSOCIE
!CORIOL*PV=DP/DX
!CORIOL*PU=-DP/DY
!PP=-CORIOL*PV*K*DXR
  DO H = 2, IDIM4J,2
     DO K = 2, IDIM4I,2
        PP(K-1, H-1) = ASPECT * (H - 2) * 0.5 * PU(K-1,H)* CORIOL
        PP(K, H-1) = ASPECT * (H - 2) * 0.5 * PU(K,H)* CORIOL
        PP(K-1, H) = ASPECT * (2 - H) * 0.5 * PU(K-1,H-1)* CORIOL
        PP(K, H) = ASPECT * (2 - H) * 0.5 * PU(K,H-1) * CORIOL
     END DO
  END DO
  RETURN  

6000 FORMAT(///,1X,'STREAMWISE FLOW AT GEOSTROPHIC BALANCE',/,1X,&
         '_______________________________',/) 
END SUBROUTINE IPRESSION_GEO


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  INISTAB
!
! DESCRIPTION
!  - creates a jet at geostrophic balance from a geopotential profile taken behind the obstacle
!______________________________________________________________________
 
SUBROUTINE INISTAB
         
  USE database
  USE parameters
  
  IMPLICIT NONE
  
!----------------------------------------------
!       L o c a l     V a r i a b l e s 
!---------------------------------------------
  REAL(KIND = SIZE_OF_REAL) :: XI0, XJ0, XDELTA, XJ1
  INTEGER :: I, J, K, H
  REAL(KIND = SIZE_OF_REAL), DIMENSION(IDIM4I,IDIM4J) :: TPP
  REAL(KIND = SIZE_OF_REAL), DIMENSION(IDIM3J) :: STRING
  REAL(KIND = SIZE_OF_REAL) :: COUPEP
  
               
! Extraction of the slice of geopotential we will extract to construct the jet
  
  OPEN(UNIT=4,ACTION="READ",FORM="UNFORMATTED")
  READ (4) NSTEP, TIMEA, TIME
  READ (4) PUA(:IDIM4I,:IDIM4J)
  READ (4) PVA(:IDIM4I,:IDIM4J)
  READ (4) PPA(:IDIM4I,:IDIM4J)
  READ (4) PU(:IDIM4I,:IDIM4J)
  READ (4) PV(:IDIM4I,:IDIM4J)
  READ (4) PP(:IDIM4I,:IDIM4J)
  
  NSTEP = 0.
  TIMEA = 0.
  TIME = 0.
  PUA(:IDIM4I,:IDIM4J) = 0.
  PVA(:IDIM4I,:IDIM4J) = 0.
  PPA(:IDIM4I,:IDIM4J) = 0.
  PU(:IDIM4I,:IDIM4J) = 0.
  PV(:IDIM4I,:IDIM4J) = 0.
  
  CALL FFTCR(PP)
  WORK(1,1:IDIM4J,1) = PP(90,1:IDIM4J)
  DO I = 1, IDIM4I
     PP(I,1:IDIM4J) = WORK(1,1:IDIM4J,1)
  END DO
  P0MAX = MAX((-MINVAL(PP(1:IDIM4I,1:IDIM4J))),(MAXVAL(PP(1:IDIM4I,1:IDIM4J))))
   
  XI0 = 512.
  XJ0 = 150.
  XJ1 = 113.
  XDELTA = 3.
  DO I = 1,IDIM4I
     DO J = 1,IDIM4J
        TPP(I,J) = (P0MAX/10000)*SIN((I - XI0)/XDELTA)*(EXP(-((I - XI0)**2 + (J - XJ0)**2)/XDELTA**2)+&
             EXP(-((I - XI0)**2 + (J - XJ1)**2)/XDELTA**2))
     END DO
  END DO
  
  PP(1:IDIM4I,1:IDIM4J) = PP(1:IDIM4I,1:IDIM4J) + TPP(1:IDIM4I,1:IDIM4J)
  
! Passage dans l'espace spectral
  CALL FFTRC(PP)
  CALL FFTRC(TPP)
! Calcul des champs de U et V correspondant a l'equilibre geostrophique
! U = -1/f * dP/dy = -1/f * i*h*P = 0
! V = 1/f * dP/dx = 1/f * i*k*P
  DO H = 2, IDIM4J, 2
     DO K = 2, IDIM4I, 2
        PU(K-1, H-1) = ASPECT * (H - 2) * 0.5 * PP(K-1,H) / CORIOL
        PU(K, H-1) = ASPECT * (H - 2) * 0.5 * PP(K,H) / CORIOL
        PU(K-1, H) = ASPECT * (2 - H) * 0.5 * PP(K-1,H-1) / CORIOL
        PU(K, H) = ASPECT * (2 - H) * 0.5 * PP(K,H-1) / CORIOL
        PV(K-1, H-1) = (2 - K) * 0.5 * TPP(K,H-1) /CORIOL
        PV(K, H-1) = (K - 2) * 0.5 * TPP(K-1, H-1) /CORIOL
        PV(K-1, H) = (2 - K) * 0.5 * TPP(K, H)/CORIOL
        PV(K, H) = (K - 2) * 0.5 * TPP(K-1, H)/CORIOL
     END DO
  END DO
   
 
END SUBROUTINE INISTAB


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  INIPENA
!
! FORMAL PARAMETERS:
!  - TYPE - masks for penalization
!
! GLOBAL PARAMETERS:
!  - KSI  
!
! DESCRIPTION:
!  - creates masks for penalization
!______________________________________________________________________

SUBROUTINE INIPENA(TYPE, NB)

  USE parameters
  USE database
  USE diagnostics, ONLY: STDERR

  IMPLICIT NONE

  INTEGER, INTENT(IN) :: TYPE, NB

  INTEGER :: I,J, I0, J0, THICKNESS, DELTA, DELTA_IN, DELTA_OUT, XS

  IF (FLOW_CASE <= 2) THEN
! The obstacle doesn't move
     I0 = IDIM3MIN/4
     J0 = IDIM3MIN/2

  ELSE IF (FLOW_CASE == 3) THEN

! The obstacle is translated at the velocity U0
     I0 = IDIM3MAX-IDIM3MIN/4-INT(U0*NB*DT/DX)
     J0 = IDIM3MIN/2
  END IF

  SELECT CASE(TYPE)

  CASE (0)
     WRITE(STDERR,*) " Error in INIPENA ! The given parameter is Zero which means no penalization !!! "
     STOP

  CASE (1) ! Cylinder

     DO J=1,IDIM3J
        DO I=1,IDIM3I
           IF ( SQRT( REAL((I-I0)*(I-I0) + (J-J0)*(J-J0)) ) &
                < DIAMETER/(2*DX) ) THEN
              KSI(I,J) = 1.
           ELSE
              KSI(I,J) = 0.
           END IF
        END DO
     END DO
     
  CASE (2) ! Smooth cylinder

! Delta is the number of space step on wich the mask will be smooth 
 
     DELTA = 2.0
     DO J=1,IDIM3J
        DO I=1,IDIM3I
           IF ( SQRT( REAL((I-I0)*(I-I0) + (J-J0)*(J-J0)) ) <= DIAMETER/DX*0.5 - DELTA) THEN
              KSI(I,J) = 1.
           ELSEIF ( ( SQRT( REAL( (I-I0)*(I-I0) + (J-J0)*(J-J0) ) ) > DIAMETER/DX*0.5 - DELTA ) .AND. &
                ( SQRT( REAL((I-I0)*(I-I0) + (J-J0)*(J-J0)) ) < DIAMETER/DX*0.5) ) THEN
              KSI(I,J) = 1.0/(1.0 + EXP( DELTA / (DIAMETER/DX*0.5 - SQRT( REAL( (I-I0)*(I-I0) + (J-J0)*(J-J0) ) )  ) &
                   + DELTA / ( DIAMETER/DX*0.5 - DELTA - SQRT( REAL( (I-I0)*(I-I0) + (J-J0)*(J-J0) ) ) ) ) )
           ELSE
              KSI(I,J) = 0.
           END IF
        END DO
     END DO

  CASE(3) ! Square
     
     DO J=1,IDIM3J
        DO I=1,IDIM3I
           IF ( ((I-I0)<DIAMETER/DX*0.5) .AND. &
                ((I-I0)>-DIAMETER/DX*0.5) .AND.&
                ((J-J0)<DIAMETER/DX*0.5) .AND. &
                ((J-J0)>-DIAMETER/DX*0.5) ) THEN
              KSI(I,J) = 1.
           ELSE
              KSI(I,J) = 0.
           END IF
        END DO
     END DO
     
  CASE(4) ! Reading from file !!!
     
     OPEN(UNIT=3, FILE="Mask_Xi",ACTION="READ",FORM="FORMATTED")
     READ (3,*) COEF_DIAM_LO
     READ (3,*) KSI
     CLOSE (3)

  CASE(5)  ! Ellipse

     DELTA_IN = DIAMETER/DX*0.5
     DELTA_OUT = 1.8*DIAMETER/DX
      DO J=1,IDIM3J
        DO I=1,IDIM3I
           IF ( SQRT( REAL((I-I0)*(I-I0)/(DELTA_IN * DELTA_IN) + (J-J0)*(J-J0)/(DELTA_OUT * DELTA_OUT)) ) &
                < 1 ) THEN
              KSI(I,J) = 1.
           ELSE
              KSI(I,J) = 0.
           END IF
        END DO
     END DO

  CASE(6) ! Flow in a cylindrical tank
     
     DO J=1,IDIM3J
        DO I=1,IDIM3I
           IF ( SQRT( REAL((I-I0)*(I-I0) + (J-J0)*(J-J0)) ) &
                < DIAMETER/DX*0.5  ) THEN
              KSI(I,J) = 0.
           ELSE
              KSI(I,J) = 1.
           END IF
        END DO
     END DO

  CASE(7) ! Test case with only sponge or walls
     
     KSI = 0.

  CASE DEFAULT

     WRITE(STDERR,*) " Error in INIPENA ! The given parameter corresponds to nothing !!! "
     STOP

  END SELECT
 
  IF (SPONGE == 1) THEN
 
! Delta_in is the number of space step on wich the sponge will be smooth for the indent flow
! Delta_out is the number of space step on wich the amplitude of the sponge decrease rapidly

     DELTA_IN = IDIM3I/4
     DELTA_OUT = 10.
     XS = IDIM3I/2 +100
     SPONGE_MASK = 0.
     DO I = XS + 1, XS + DELTA_IN - 1
        SPONGE_MASK(I,1:IDIM3J)= 1.0/(1.0+EXP(1.0*DELTA_IN/(1.0*I - XS)+ 1.0*DELTA_IN/(1.0*I - (XS + DELTA_IN))))
     END DO
     DO I = XS + DELTA_IN ,IDIM3I - DELTA_OUT
        SPONGE_MASK(I,1:IDIM3J) = 1.
     END DO
     DO I = IDIM3I-DELTA_OUT + 1, IDIM3I-1
        SPONGE_MASK(I,1:IDIM3J) = 1.0/(1.0+EXP(1.0*DELTA_OUT/(IDIM3I - DELTA_OUT - I) + 1.0*DELTA_OUT / (IDIM3I-I)))
     END DO
  
  ENDIF
  
  IF (WALLS /= 0) THEN

! Thickness is the number of space step for the lateral walls

     THICKNESS = 5
     WALL_MASK=0.
     WALL_MASK(1:THICKNESS, 1:IDIM3J)=1.
     WALL_MASK((IDIM3I-THICKNESS+1):IDIM3I , 1:IDIM3J)=1.

  ENDIF
  
END SUBROUTINE INIPENA 


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  INROTA 
!
! FORMAL PARAMETERS:
!  - POM - vorticity fluctuations
!
! GLOBAL PARAMETER:
!  - PU, PV, PO  - velocity and vorticity fields
! DESCRIPTION:
!  - creates an initial divergence-free field with random vorticity
!  - determines the fluctuations
!______________________________________________________________________

SUBROUTINE INROTA(POM)

  USE database
  USE parameters
  USE diagnostics, ONLY: OUTPUT1

  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL), INTENT(IN) :: POM
     
! creating random velocity field
  CALL IRANDOM (PO, POM) 
  CALL OVELOCITY (PO, PU, PV)
  
  IF ( INCOMP > 0 ) THEN
     CALL PRESSURE_CORRECTION( 0 )
! This messes up the vorticity field which has to be recreated;
     CALL OROT ( PU, PV, PO )
  ENDIF

! Going to real space to calculate extremal fluctuations (need V0MAX);
  CALL FFTCR (PU) 
  CALL FFTCR (PV) 
  CALL FFTCR (PO) 
  IF ( INCOMP == 0 ) CALL FFTCR (PP)

  CALL DFLUCT

  CALL FFTRC (PU) 
  CALL FFTRC (PV) 
  CALL FFTRC (PO) 
  IF ( INCOMP == 0 ) CALL FFTRC (PP)          

  IF ( VERBOSE > 1 ) WRITE(OUTPUT1,*)'  Initializing with an div-free random vorticity field'

END SUBROUTINE INROTA


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  INBAL
!
! PARAMETERS:
!  - NONE
!
! DESCRIPTION:
!  - creates an initial field in a balanced equilibrium
!  - determines the fluctuations
!______________________________________________________________________

SUBROUTINE INBAL 

  USE database
  USE parameters
  USE diagnostics, ONLY: OUTPUT1

  IMPLICIT NONE

  INTEGER :: K, H 
  REAL(KIND=SIZE_OF_REAL) :: PMMAX, PONDER, T 

!_______________________________________________________________________
!     DERNIERE MODIFICATION INBAL: NOVEMBRE 85
!_______________________________________________________________________
!
!  COMMUNS EN ENTREE:   PM         FLUCTUATIONS PRESSION (NON REDUITES)
!                       P0FORCE    RESSION FORCEE (REDUITE)
!                       U0         ITESSE FORCEE (REDUITE)
!                       CORIOL     PARAMETRE DE CORIOLIS
! 
!  COMMUNS EN SORTIE:   PP         FLUCTUATIONS DE PRESSION (REDUITES)
!                       PU         VITESSE EN ABSCISSE (REDUITE)
!                       PV         VITESSE EN ORDONNEE (REDUITE)
!
!_______________________________________________________________________
!
!  ECOULEMENT EN EQUILIBRE BALANCE
!  _______________________________
!  ON INITIALISE AVEC UN CHAMP DE PRESSION ALEATOIRE, ON CALCULE LE
!  CHAMP DE VITESSE EN EQUILIBRE GEOSTROPHIQUE CORRESPONDANT, PUIS
!  ON RECALCULE LE CHAMP DE PRESSION DE FA\ON A ANNULER LA DERIVEE
!  TEMPORELLE DE LA DIVERGENCE EN RESOLVANT:
!  D2(P+(V2/2))=ROT((O+CORIOL)*V)
  IF ( VERBOSE > 2 ) WRITE (OUTPUT1, 6000) 

!  TIRAGE D'UN CHAMP DE PRESSION ALEATOIRE
!  _______________________________________

  PM = P0*FLUCTP/100
  CALL IRANDOM (PP, PM) 

!  RETOUR DANS L'ESPACE PHYSIQUE
  CALL FFTCR (PP) 

!  PONDERATION DES FLUCTUATIONS DE PRESSION
!  ________________________________________
!  RECHERCHE DE LA VALEUR MAXIMALE DES FLUCTUATIONS PMAX
!  ET CALCUL DU COEFFICIENT DE PONDERATION PONDER PERMETTANT DE RAMENER
!  LES FLUCTUATIONS DE PRESSION A UNE VALEUR MAXIMALE EGALE A CFLUCTP*P0
!  PONDER=(CFLUCTP*P0)/PMAX

  PMMAX = MAXVAL(PP(1:IDIM3I,1:IDIM3J))
  PONDER = (FLUCTP*P0)/(PMMAX*100) 
  PP(1:IDIM4I,1:IDIM4J) = PP(1:IDIM4I,1:IDIM4J)*PONDER 

!  RETOUR DANS L'ESPACE SPECTRAL
  CALL FFTRC (PP) 

!  CALCUL DU CHAMP DE VITESSE EN EQUILIBRE GEOSTROPHIQUE
!  _____________________________________________________
!  AVEC LE CHAMP DE PRESSION , I.E. TEL QUE:
!  U=(-1/CORIOL)*(DP/DY)
!  V=(+1/CORIOL)*(DP/DX)
!  EN SPECTRAL  U=-I*H*P/CORIOL
!  EN SPECTRAL  V=+I*K*P/CORIOL
!  OPERATEUR DE DERIVATION NON REDUIT
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  DO H = 2, IDIM4J, 2 
     DO K = 2, IDIM4I, 2 
        PU(K-1,H-1) = ASPECT*(H - 2)*PP(K-1,H)*0.5/CORIOL 
        PU(K,H-1) = ASPECT*(H - 2)*PP(K,H)*0.5/CORIOL 
     END DO
     DO K = 2, IDIM4I, 2 
        PU(K-1,H) = ASPECT*(2 - H)*PP(K-1,H-1)*0.5/CORIOL 
        PU(K,H) = ASPECT*(2 - H)*PP(K,H-1)*0.5/CORIOL 
     END DO
  END DO
  DO H = 2, IDIM4J, 2 
     DO K = 2, IDIM4I, 2 
        PV(K-1,H-1) = (2 - K)*PP(K,H-1)*0.5/CORIOL 
        PV(K,H-1) = (K - 2)*PP(K-1,H-1)*0.5/CORIOL 
     END DO
     DO K = 2, IDIM4I, 2 
        PV(K-1,H) = (2 - K)*PP(K,H)*0.5/CORIOL 
        PV(K,H) = (K - 2)*PP(K-1,H)*0.5/CORIOL 
     END DO
  END DO

!  CALCUL DE LA VORTICITE
!  _______________________
  CALL OROT (PU, PV, PO) 
  PO(1,1) = PO(1,1) + CORIOL 

!  RETOUR DANS L'ESPACE PHYSIQUE
  CALL FFTCR (PU) 
  CALL FFTCR (PV) 
  CALL FFTCR (PO) 

!  CALCUL DE ((O+CORIOL)*V)
!  ________________________
!  PRODUITS (O+CORIOL)*U ET (O+CORIOL)*V RANGES DANS WORK ET PF
  PO(1:IDIM4I,1:IDIM4J) = PO(1:IDIM4I,1:IDIM4J)*PU(1:IDIM4I,1:IDIM4J) 
  PF(1:IDIM4I,1:IDIM4J) = PO(1:IDIM4I,1:IDIM4J)*PV(1:IDIM4I,1:IDIM4J) 

!  RETOUR DANS L'ESPACE SPECTRAL
  CALL FFTRC (PO) 
  CALL FFTRC (PF) 

!  CALCUL DE ROT((O+CORIOL)*V)
!  __________________________
  CALL OROT (PO, PF, WORK) 

!  RESOLUTION DE L'EQUATION DE POISSON
!  ___________________________________
  CALL OLAPINV (WORK, PP) 

  IF (IFORCE == 1) THEN 
!  FORCAGE ZONAL GEOSTROPHIQUE
!  ___________________________
     PP(1,1) = P0FORCE 
     PP(1,7) = - U0 / 6.0 * CORIOL 
     PP(1,8) = 0. 
     PU(1,7) = 0. 
     PU(1,8) = U0 / 2.0 
     PV(1,7) = 0. 
     PV(1,8) = 0. 
  ENDIF

!  RETOUR DANS L'ESPACE PHYSIQUE
  CALL FFTCR (PP) 

!  CALCUL DE P-V2/2
  PP(1:IDIM4I,1:IDIM4J) = PP(1:IDIM4I,1:IDIM4J) - (PU(1:IDIM4I,1:IDIM4J)**2+PV(1:IDIM4I,1:IDIM4J)**2)*0.5 

!  ECRITURE DES FLUCTUATIONS
!  -------------------------
  CALL DFLUCT 

!  RETOUR DANS L'ESPACE SPECTRAL
  CALL FFTRC (PU) 
  CALL FFTRC (PV) 
  CALL FFTRC (PP) 

  MEAN_PRESS=PP(1,1)

  RETURN  
6000 FORMAT(///,1X,'ECOULEMENT EN EQUILIBRE BALANCE',/,1X,&
         '_______________________________',/) 
END SUBROUTINE INBAL


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  IRANDOM
!
! FORMAL PARAMETERS:
!  - FM  - fluctuation level 
!  - F - random field to be created (in Fourier space)
!
! GLOBAL PARAMETERS:
!  - KEXCIT  - excited wavenumber
!  - KCUTOFF   - cut-off wavenumber
!
! DESCRIPTION:
!  - creates the random field FM with prescribed spectral slope and 
!    fluctuation level specified in F
!______________________________________________________________________


SUBROUTINE IRANDOM(F, FM) 

  USE parameters
  USE database

  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL) , INTENT(IN) :: FM 
  REAL(KIND=SIZE_OF_REAL) , INTENT(OUT) :: F(IDIM4I,IDIM4J) 

  INTEGER :: K, KEXMIN, KEXMAX, KEXCIT_2, IT1, IT2, H 
  REAL(KIND=SIZE_OF_REAL) :: RK2, T, X1, X2, A1, B1, A2, B2, K2

  KEXMIN = 1 
  KEXMAX = KCUTOFF 
  KEXCIT_2 = KEXCIT*KEXCIT 

!  Fields with a random phase
!  _______________________________
!  F=F0+FM*KEXCIT*COS(X+Y)
!    +sum from K=0      to KEXCIT DE(FM*K**(+3/2)*COS(K SCAL X+PHI)
!    +sum from K=KEXCIT to KMAX   DE(FM*K**(-3/2)*COS(K SCAL X+PHI)
!  FM*KEXCIT*COS(X+Y)  fluctuations with spectral amplitude
!                                    COS(K-1,H-1)=-1/4
!                                    COS(K-1,H  )=0           0   -1/4
!                                    COS(K  ,H  )=+1/4       +1/4    0
!                                    COS(K  ,H-1)=0
!  SOMME SUR K         HARMONIQUES AJOUTEES POUR REMPLIR LE SPECTRE
!  PHI                 PHASE ALEATOIRE PERMETTANT DE GENERER DES SINUS
!                      POUR AVOIR UN SPECTRE CONTINU
!                      EN SPECTRAL   SIN(K-1,H-1)=0
!                                    SIN(K-1,H  )=-1/4      -1/4    0
!                                    SIN(K  ,H  )=0          0     -1/4
!                                    SIN(K  ,H-1)=+1/4
!  IT1 ET IT2      CORRECTIONS A EFFECTUER SUR LES FRONTIERES DU DOMAINE
!  IT1*IT2=1*1=1   A L'ORIGINE
!  IT1*IT2=1*2=2   SUR LA FRONTIERE 0,K
!  IT1*IT2=2*1=2   SUR LA FRONTIERE 0,H
!  IT1*IT2=2*2=4   DANS LE DOMAINE

  IT1 = 1 
  DO K = 2*KEXMIN, 2*KEXMAX, 2 
     IT2 = 1 
     DO H = 2*KEXMIN, 2*KEXMAX, 2 
        K2 = ((K - 2)*(K - 2) + ASPECT2*(H - 2)*(H - 2))/4 
        RK2 = K2/FLOAT(KEXCIT_2) 
        IF (K2 - KEXCIT_2 < 0) THEN 

!  from  KEXMIN to KEXCIT
           T = FM*0.25*RK2**(3./2.)
!CDIR IEXPAND 
           CALL RANDOM_NUMBER (X1) 
           X1 = 2.*PI*X1 
!CDIR IEXPAND 
           CALL RANDOM_NUMBER (X2) 
           X2 = 2.*PI*X2 
           A1 = T*COS(X1) 
           B1 = T*SIN(X1) 
           A2 = T*COS(X2) 
           B2 = T*SIN(X2) 
           SELECT CASE (IT1)  
           CASE DEFAULT 
              F(K-1,H-1) = A1*0.5 
              F(K-1,H) = B1*0.5 
              GO TO 77781 
           CASE (2)  
              SELECT CASE (IT2)  
              CASE DEFAULT 
                 F(K-1,H-1) = A1*0.5 
                 F(K,H-1) = B1*0.5 
                 GO TO 77781 
              CASE (2)  
                 F(K-1,H-1) = A1 + A2 
                 F(K-1,H) = (-B1) + B2 
                 F(K,H) = (-A1) + A2 
                 F(K,H-1) = (-B1) - B2 
                 GO TO 77781 
              END SELECT
           END SELECT
        ELSE          
!  from KEXCIT to KEXMAX
           T = FM*0.25*RK2**(-3./2.) 
!CDIR IEXPAND 
           CALL RANDOM_NUMBER (X1) 
           X1 = 2.*PI*X1 
!CDIR IEXPAND 
           CALL RANDOM_NUMBER (X2) 
           X2 = 2.*PI*X2 
           A1 = T*COS(X1) 
           B1 = T*SIN(X1) 
           A2 = T*COS(X2) 
           B2 = T*SIN(X2) 
           SELECT CASE (IT1)  
           CASE DEFAULT 
              F(K-1,H-1) = A1*0.5 
              F(K-1,H) = B1*0.5 
           CASE (2)  
              SELECT CASE (IT2)  
              CASE DEFAULT 
                 F(K-1,H-1) = A1*0.5 
                 F(K,H-1) = B1*0.5 
              CASE (2)  
                 F(K-1,H-1) = A1 + A2 
                 F(K-1,H) = (-B1) + B2 
                 F(K,H) = (-A1) + A2 
                 F(K,H-1) = (-B1) - B2 
              END SELECT
           END SELECT
        END IF
77781   CONTINUE 
        IT2 = 2 
     END DO
     IT1 = 2 

! the imag part of the first coeffcient is zero (real field)
     F(K,2) = 0. 
     F(2,K) = 0. 
  END DO
  RETURN  
END SUBROUTINE IRANDOM


!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  DFLUCT
!
! FORMAL PARAMETERS:
!  - NONE
!
! GLOBAL PARAMETERS:
!  - PU, PV, PP   
!  - CSON, P0
!  - P0MAX, VOMAX
!
! DESCRIPTION:
!  - calculates the maximum values of velocity and pressure fluctuations
!______________________________________________________________________

SUBROUTINE DFLUCT 
  
  USE database
  USE parameters
  USE diagnostics, ONLY: UMIN, UMAX, VMIN, VMAX, PMIN, PMAX
  IMPLICIT NONE

 UMIN = MINVAL(PU(1:IDIM3I,1:IDIM3J))
 UMAX = MAXVAL(PU(1:IDIM3I,1:IDIM3J))
 VMIN = MINVAL(PV(1:IDIM3I,1:IDIM3J))
 VMAX = MAXVAL(PV(1:IDIM3I,1:IDIM3J))
 PMIN = MINVAL(PP(1:IDIM3I,1:IDIM3J))
 PMAX = MAXVAL(PP(1:IDIM3I,1:IDIM3J))
 WORK(1:IDIM4I,1:IDIM4J,1) = SQRT(PU(1:IDIM4I,1:IDIM4J)**2+PV(1:IDIM4I,1:IDIM4J)**2) 
 V0MAX = MAXVAL(WORK(1:IDIM3I,1:IDIM3J,1))
 P0MAX = MAX((-PMIN),PMAX)

END SUBROUTINE DFLUCT


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  DINVAR
!
! FORMAL PARAMETERS:
!  - E     - energy
!  - ZP    - {potential) enstrophy
!
! GLOBAL PARAMETERS:
!  COMMUNS EN ENTREE:   ERREUR   VALEUR DES INVARIANTS AU-DEL@ DE
!                                LAQUELLE ON CONSIDERE QUE LE MODELE
!                                DIVERGE
!                       PU       VITESSE EN ABSCISSE (PHYSIQUE)
!                       PV       VITESSE EN ORDONNEE (PHYSIQUE)
!                       PP       PRESSION (PHYSIQUE)
!                       PO       VORTICITE ABSOLUE (PHYSIQUE)
!                       NMPOINT  NOMBRE DES POINTS DE GRILLE
!                       CORIOL   PARAMETRE DE CORIOLIS
!                       V0MAX    VITESSE MAXIMALE (MODULE)
!
!  COMMUNS EN SORTIE:   ETOT     ENERGIE TOTALE AU PAS N
!                       ZPOT     ENSTROPHIE POTENTIELLE AU PAS N
!                       NTIME    PAS DE TEMPS OU L'ON MEMORISE E ET ZP
!                       ETIME    DONNE L'EVOLUTION TEMPORELLE DE ETOT
!                       ZPTIME   DONNE L'EVOLUTION TEMPORELLE DE ZPOT
!
! DESCRIPTION:
!  - calculates the quadratic/cubic invariants: energy 
!  - and (potential) enstrophy
!______________________________________________________________________

SUBROUTINE DINVAR(E, ZP) 

  USE database
  USE parameters
  USE diagnostics
 
  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL), INTENT(OUT) :: E, ZP

!  EKIN=double sum (1/2((P0+P')*(U2+V2))) DXDY / Area
!  EPOT=double sum (1/2(P'2)) DXDY / Area
! potential energy due to mean pressure removed
!  ELIN=double sum (1/2(P0*(P'+U2+V2))) DXDY / Area
!      =double sum (1/2(P0*(U2+V2))) DXDY / Area + EPOT
!  ETOT=EKIN+EPOT
!  ZTOT=double sum (1/2(O2)) DXDY / Area
!  ZPOT=double sum (1/2(O2/P)) DXDY / Area
!  ZLIN=double sum (1/2(O2/P0)) DXDY / Area

! The invariants EPOT, ELIN, ZPOT and ZLIN are not relevant in 
! incompressible case;

 WORK(1:IDIM4I,1:IDIM4J,1) = PU(1:IDIM4I,1:IDIM4J)**2+PV(1:IDIM4I,1:IDIM4J)**2
 IF ( INCOMP == 0 ) THEN
    ELIN = MEAN_PRESS*SUM(WORK(1:IDIM3I,1:IDIM3J,1))
    EPOT = SUM(PP(1:IDIM3I,1:IDIM3J)**2)
    EKIN = SUM((MEAN_PRESS+PP(1:IDIM3I,1:IDIM3J))*WORK(1:IDIM3I,1:IDIM3J,1))
    ZTOT = SUM((PO(1:IDIM3I,1:IDIM3J)-CORIOL)**2)
    ZPOT = SUM(PO(1:IDIM3I,1:IDIM3J)**2/(MEAN_PRESS+PP(1:IDIM3I,1:IDIM3J)))
    ZLIN = SUM((PO(1:IDIM3I,1:IDIM3J)-CORIOL-CORIOL*PP(1:IDIM3I,1:IDIM3J)/MEAN_PRESS)**2)
 ELSE
    EKIN = SUM(WORK(1:IDIM3I,1:IDIM3J,1))
    ZTOT = SUM(PO(1:IDIM3I,1:IDIM3J)**2)         
 ENDIF

  EKIN = 0.5*EKIN/FLOAT(NMPOINT) 
  ZTOT = 0.5*ZTOT/FLOAT(NMPOINT) 
  IF ( INCOMP == 0 ) THEN
     EPOT = 0.5*EPOT/FLOAT(NMPOINT) 
     ELIN = 0.5*ELIN/FLOAT(NMPOINT) + EPOT 
     ZPOT = 0.5*ZPOT/FLOAT(NMPOINT)
     ZLIN = 0.5*ZLIN/FLOAT(NMPOINT)
  ELSE
     EPOT = 0.0;   ELIN = 0.0;   ZPOT = 0.0;   ZLIN = 0.0
  ENDIF
  ETOT = EKIN + EPOT 

  E = ETOT 
! Potential or total enstrophy returned as ZP depending on the case;
  IF ( INCOMP == 0 ) THEN
     ZP = ZPOT 
  ELSE
     ZP = ZTOT
  ENDIF

END SUBROUTINE DINVAR


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  TPENA
!
! FORMAL PARAMETERS:
!  - TYPE - penalization type
!
! GLOBAL PARAMETERS:
!  - PUAA, PFAA, PO, PF
!
! DESCRIPTION:
!  - calculates the penalization terms
!______________________________________________________________________

SUBROUTINE TPENA(TYPE)
  
  USE parameters
  USE database
  IMPLICIT NONE

  INTEGER, INTENT(IN) :: TYPE
  INTEGER :: I,J

  CALL FFTCR(PF)
  CALL FFTCR(PO)
  CALL FFTCR(PUAA)
  CALL FFTCR(PVAA)
  IF (SPONGE == 1 ) THEN
     CALL FFTCR(PP)
     CALL FFTCR(PPAA)
  ENDIF
     
  IF (TYPE == 1) THEN              ! The obstacle is at rest
     IF ((WALLS == 0) .AND. (SPONGE == 0))THEN
        PF = PF * KSI 
        PO = PO * KSI
        PUAA = PUAA * KSI * KSI
        PVAA = PVAA * KSI * KSI
     ELSE IF ((WALLS == 0) .AND. (SPONGE == 1)) THEN
        PF = PF * KSI * SPONGE_MASK
        PO = PO * KSI * SPONGE_MASK
        PP = PP * SPONGE_MASK
        PUAA = PUAA * KSI * KSI * SPONGE_MASK * SPONGE_MASK
        PVAA = PVAA * KSI * KSI * SPONGE_MASK * SPONGE_MASK
        PPAA = PPAA * SPONGE_MASK * SPONGE_MASK
        DO I = 1,IDIM4I
           DO J = 1,IDIM4J
              GEOP_FORCE(I,J) = P0 * ( 1.0 - SPONGE_MASK(I,J) * SPONGE_MASK(I,J))
              VELO_FORCE(I,J) = U0 * ( 1.0 - SPONGE_MASK(I,J) * SPONGE_MASK(I,J))
           END DO
        END DO
     ELSE IF ((WALLS == 1) .AND. (SPONGE == 0)) THEN     ! No-slip boundary conditions on the walls
        PF = PF * KSI * WALL_MASK
        PO = PO * KSI * WALL_MASK
        PUAA = PUAA * KSI * KSI * WALL_MASK * WALL_MASK
        PVAA = PVAA * KSI * KSI * WALL_MASK * WALL_MASK
     ELSE IF ((WALLS == 1) .AND. (SPONGE == 1)) THEN
        PF = PF * KSI * WALL_MASK * SPONGE_MASK
        PO = PO * KSI * WALL_MASK * SPONGE_MASK
        PP = PP * SPONGE_MASK
        PUAA = PUAA * KSI * KSI * WALL_MASK * WALL_MASK * SPONGE_MASK * SPONGE_MASK
        PVAA = PVAA * KSI * KSI * WALL_MASK * WALL_MASK *SPONGE_MASK * SPONGE_MASK
        PPAA = PPAA * SPONGE_MASK * SPONGE_MASK
        DO I = 1,IDIM4I
           DO J = 1,IDIM4J
              GEOP_FORCE(I,J) = P0 * ( 1.0 - SPONGE_MASK(I,J) * SPONGE_MASK(I,J))
              VELO_FORCE(I,J) = U0 * ( 1.0 - SPONGE_MASK(I,J) * SPONGE_MASK(I,J))
           END DO
        END DO
     END IF
     
  ELSE IF (TYPE == 2) THEN
     IF ((WALLS == 0) .AND. (SPONGE == 1)) THEN
        PF = PF * KSI * SPONGE_MASK
        PO = PO * KSI * SPONGE_MASK
        PUAA = PUAA * KSI * KSI * SPONGE_MASK * SPONGE_MASK
        PVAA = PVAA * KSI * KSI * SPONGE_MASK * SPONGE_MASK        
     END IF
        
  ELSE IF (TYPE == 3) THEN
     IF (WALLS == 0) THEN
        PF = PF * KSI 
        PO = (PO + VELO_FORCE) * KSI
        PUAA = PUAA * KSI * KSI
        PVAA = PVAA * KSI * KSI
     ELSE IF (WALLS == 1) THEN
        PF = PF * KSI * WALL_MASK
        PO = (PO + VELO_FORCE) * KSI * WALL_MASK
        PUAA = PUAA * KSI * KSI * WALL_MASK * WALL_MASK
        PVAA = PVAA * KSI * KSI * WALL_MASK * WALL_MASK
     END IF
  END IF  

  CALL FFTRC(PF)
  CALL FFTRC(PO)
  CALL FFTRC(PUAA)
  CALL FFTRC(PVAA)
  IF (SPONGE == 1) THEN
     CALL FFTRC(PP)
     CALL FFTRC(PPAA)
     CALL FFTRC(GEOP_FORCE)
     CALL FFTRC(VELO_FORCE)
  ENDIF

END SUBROUTINE TPENA


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  TLEAPFROG (Bartosz Protas, 18/12/2003)
!
! FORMAL PARAMETERS:
!  - NONE
!
! GLOBAL PARAMETERS:
!  - PU, PV, PP, PUAA, PVAA, PPAA, PO, PF
!  - DISSIP, DISSIP2, DT
!
! DESCRIPTION:
!  - using leapfrog time stepping, advances the system one step in time
!______________________________________________________________________

SUBROUTINE TLEAPFRO

  USE parameters
  USE database
  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL) :: DDT 
  INTEGER :: I

!  leapfrog scheme
!  _______________
!  F(N+1)=F(N-1)*EXP(-2*NU*K**(2*DISSIPATION_EXPONENT)*DT)+D(F(N))/DT*EXP(-NU*K**(2*DISSIPATION_EXPONENT)*DT)*2DT
!  F(N-1)   field at the time instant N-1  (AA)
!  F(N+1)   field at the time instant N+1  (  )
!  DU(N)/DT in PF
!  DV(N)/DT in PO
!  DP(N)/DT in PP

  DDT = 2.*DT 
!D=EXP(-NU*K**(2*DISSIPATION_EXPONENT)*DT)=DISSIP(K)
!$OMP PARALLEL DO 
  DO I=1,IDIM4J
     PV(1:IDIM4I,I) = DISSIP(1:IDIM4I,I)*(DISSIP(1:IDIM4I,I)*PVAA(1:IDIM4I,I)+PO(1:IDIM4I,I)*DDT) 
  ENDDO
  IF (SPONGE == 1) THEN
!$OMP PARALLEL DO 
     DO I=1,IDIM4J
        PU(1:IDIM4I,I) = DISSIP(1:IDIM4I,I)*(DISSIP(1:IDIM4I,I)*PUAA(1:IDIM4I,I)+&
             PF(1:IDIM4I,I)*DDT+VELO_FORCE(1:IDIM4I,I)) 
        PP(1:IDIM4I,I) = PPAA(1:IDIM4I,I)+PP(1:IDIM4I,I)*DDT+GEOP_FORCE(1:IDIM4I,I)
     ENDDO
  ELSE
!$OMP PARALLEL DO 
     DO I=1,IDIM4J
        PU(1:IDIM4I,I) = DISSIP(1:IDIM4I,I)*(DISSIP(1:IDIM4I,I)*PUAA(1:IDIM4I,I)+PF(1:IDIM4I,I)*DDT) 
        PP(1:IDIM4I,I) = PPAA(1:IDIM4I,I)+PP(1:IDIM4I,I)*DDT
     ENDDO
  END IF

END SUBROUTINE TLEAPFRO



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  TADAMS_BASHFORTH (Bartosz Protas, 18/12/2003)
!
! FORMAL PARAMETERS:
!  - NONE
!
! GLOBAL PARAMETERS:
!  - PU, PV, PP, PUAA, PVAA, PPAA, PO, PF
!  - DISSIP, DISSIP2, DT
!
! DESCRIPTION:
!  - using AB time stepping, advances the system one step in time
!______________________________________________________________________

SUBROUTINE TADAMS_BASHFORTH

  USE parameters
  USE database
  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL) :: DT2 

  DT2 = DT / 2.0 
  PU(1:IDIM4I,1:IDIM4J) = DISSIP(1:IDIM4I,1:IDIM4J)*(DISSIP(1:IDIM4I,1:IDIM4J)*PUA(1:IDIM4I,1:IDIM4J)+ & 
       (3.0*PF(1:IDIM4I,1:IDIM4J)-PUAA(1:IDIM4I,1:IDIM4J))*DT2) 
  PV(1:IDIM4I,1:IDIM4J) = DISSIP(1:IDIM4I,1:IDIM4J)*(DISSIP(1:IDIM4I,1:IDIM4J)*PVA(1:IDIM4I,1:IDIM4J)+ & 
       (3.0*PO(1:IDIM4I,1:IDIM4J)-PVAA(1:IDIM4I,1:IDIM4J))*DT2) 
  PUAA(1:IDIM4I,1:IDIM4J) = PF(1:IDIM4I,1:IDIM4J)
  PVAA(1:IDIM4I,1:IDIM4J) = PO(1:IDIM4I,1:IDIM4J)

END SUBROUTINE TADAMS_BASHFORTH



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  TDISSIP 
!
! FORMAL PARAMETERS:
!  - NONE
!
! GLOBAL PARAMETERS:
!  - WORK, NU, DT
!  - DISSIP
!
! DESCRIPTION:
!  - calculates the dissipation matrix for the full time step
!______________________________________________________________________

SUBROUTINE TDISSIP 

  USE database
  USE parameters
  IMPLICIT NONE

  INTEGER :: K,  H
  REAL(KIND=SIZE_OF_REAL) :: WORKVAL, K2
  
  DO H = 2, IDIM4J, 2 
     DO K = 2, IDIM4I, 2 
        K2 = ((K - 2)*(K - 2) + ASPECT2*(H - 2)*(H - 2))/4
        WORKVAL = EXP((-NU*K2**DISSIPATION_EXPONENT*DT))
        DISSIP(K-1,H-1) = WORKVAL
        DISSIP(K-1,H) = WORKVAL
        DISSIP(K,H) = WORKVAL
        DISSIP(K,H-1) = WORKVAL
     END DO
  END DO
  
  DISSIP(1,1) = 1.0
  DISSIP(1,2) = 1.0
  DISSIP(2,2) = 1.0 
  DISSIP(2,1) = 1.0 
 
  RETURN  
END SUBROUTINE TDISSIP


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  TDISSIP2 
!
! FORMAL PARAMETERS:
!  - NONE
!
! GLOBAL PARAMETERS:
!  - WORK, NU, DT
!  - DISSIP
!
! DESCRIPTION:
!  - calculates the dissipation matrix for half the time step
!______________________________________________________________________

SUBROUTINE TDISSIP2 

  USE database
  USE parameters
  IMPLICIT NONE

  INTEGER :: K, H
  REAL(KIND=SIZE_OF_REAL) :: WORKVAL, DT2, K2

  DT2 = DT / 2.0
     
  DO H = 2, IDIM4J, 2 
     DO K = 2, IDIM4I, 2 
        K2 = ((K - 2)*(K - 2) + ASPECT2*(H - 2)*(H - 2))/4
        WORKVAL = EXP((-NU*K2**DISSIPATION_EXPONENT*DT2))
        DISSIP2(K-1,H-1) = WORKVAL
        DISSIP2(K-1,H) = WORKVAL
        DISSIP2(K,H) = WORKVAL
        DISSIP2(K,H-1) = WORKVAL
     END DO
  END DO
  
  DISSIP2(1,1) = 1.0 
  DISSIP2(1,2) = 1.0 
  DISSIP2(2,2) = 1.0 
  DISSIP2(2,1) = 1.0 
  
  RETURN  

END SUBROUTINE TDISSIP2



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  MPROGE
!
! FORMAL PARAMETERS:
!  - U, V, P - velocity components and pressure
!  - F0      - mean pressure
!  - C0      - Coriolis number
!
! GLOBAL PARAMETERS:
!  - NONE
!
! DESCRIPTION:
!  - removes small-scale inertio-gravity waves
!______________________________________________________________________

SUBROUTINE MPROGE(U, V, P, F0, CO) 

  USE parameters
  USE database
  USE diagnostics, ONLY: OUTPUT1

  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL) , INTENT(IN) :: F0 
  REAL(KIND=SIZE_OF_REAL) , INTENT(IN) :: CO 
  COMPLEX(KIND=SIZE_OF_COMPLEX) :: U(IDIM2I,IDIM4J) 
  COMPLEX(KIND=SIZE_OF_COMPLEX) :: V(IDIM2I,IDIM4J) 
  COMPLEX(KIND=SIZE_OF_COMPLEX) :: P(IDIM2I,IDIM4J) 

  INTEGER :: KX, KY
  REAL(KIND=SIZE_OF_REAL) :: K2
  REAL(KIND=SIZE_OF_REAL) :: A 
  COMPLEX(KIND=SIZE_OF_COMPLEX):: PK, PSK, IMAG 


  IF ( VERBOSE > 1 )  WRITE(OUTPUT1,*)'  Eliminating IG modes'     

  IMAG = (0.,1.) 
  CALL MDECOD (U) 
  CALL MDECOD (V) 
  CALL MDECOD (P) 
  DO KY = 1, IDIM2J 
     DO KX = 1, IDIM2I 
        K2 = (KX - 1)*(KX - 1) + ASPECT2*(KY - 1)*(KY - 1) 
        A = (CO*CO + K2*F0)**0.5 
        PK = (CO*P(KX,KY+KY-1)+IMAG*F0*((KY-1)*U(KX,KY+KY-1)-(KX-1)*V(KX,KY+KY-1)))/A 
        PSK = (CO*P(KX,KY+KY)+IMAG*F0*((-(KY-1)*U(KX,KY+KY))-(KX-1)*V(KX,KY+KY)))/A 
        P(KX,KY+KY-1) = PK*CO/A 
        P(KX,KY+KY) = PSK*CO/A 
        U(KX,KY+KY-1) = PK*IMAG*(1 - KY)/A 
        U(KX,KY+KY) = PSK*IMAG*(KY - 1)/A 
        V(KX,KY+KY-1) = PK*IMAG*(KX - 1)/A 
        V(KX,KY+KY) = PSK*IMAG*(KX - 1)/A 
     END DO
  END DO
  CALL MCOD (U) 
  CALL MCOD (V) 
  CALL MCOD (P) 
  RETURN  
END SUBROUTINE MPROGE



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  MCHGT
!
! FORMAL PARAMETERS:
!  - U, V - v_x / v_y, on exit vorticity and divergence
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - replaces the two velocity components with the vorticity and 
!    divergence of the field
!______________________________________________________________________

SUBROUTINE MCHGT(U, V)

  USE parameters, ONLY:ASPECT,ASPECT2
  USE database
  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL) , INTENT(INOUT) :: U(IDIM4I,IDIM4J) 
  REAL(KIND=SIZE_OF_REAL) , INTENT(INOUT) :: V(IDIM4I,IDIM4J) 
  
  INTEGER :: KX, KY 
  REAL(KIND=SIZE_OF_REAL) :: TX, TY, D1, D2, D3, D4, V1, V2, V3, V4 

  DO KY = 2, IDIM4J, 2 
     DO KX = 2, IDIM4I, 2 
        TX = (KX - 2)*0.5 
        TY = ASPECT*(KY - 2)*0.5 
        D1 = (-TX*U(KX,KY-1)) - TY*V(KX-1,KY) 
        D2 = TX*U(KX-1,KY-1) - TY*V(KX,KY) 
        D3 = (-TX*U(KX,KY)) + TY*V(KX-1,KY-1) 
        D4 = TX*U(KX-1,KY) + TY*V(KX,KY-1) 
        V1 = TY*U(KX-1,KY) - TX*V(KX,KY-1) 
        V2 = TY*U(KX,KY) + TX*V(KX-1,KY-1) 
        V3 = (-TY*U(KX-1,KY-1)) - TX*V(KX,KY) 
        V4 = (-TY*U(KX,KY-1)) + TX*V(KX-1,KY) 
        U(KX-1,KY-1) = D1 
        V(KX-1,KY-1) = V1 
        U(KX,KY-1) = D2 
        V(KX,KY-1) = V2 
        U(KX-1,KY) = D3 
        V(KX-1,KY) = V3 
        U(KX,KY) = D4 
        V(KX,KY) = V4 
     END DO
  END DO
  RETURN  
END SUBROUTINE MCHGT



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  MCHGTINV
!
! FORMAL PARAMETERS:
!  - U, V - vorticity and divergence, on exit v_x / v_y
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - replaces the vorticity and divergence of a field with the two
!    velocity components
!______________________________________________________________________

SUBROUTINE MCHGTINV(U, V) 
  
  USE parameters, ONLY:ASPECT, ASPECT2
  USE database
  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL) , INTENT(INOUT) :: U(IDIM4I,IDIM4J) 
  REAL(KIND=SIZE_OF_REAL) , INTENT(INOUT) :: V(IDIM4I,IDIM4J) 

  INTEGER :: KX, KY
  REAL(KIND=SIZE_OF_REAL) :: K2 
  REAL(KIND=SIZE_OF_REAL) :: TX, TY, U1, U2, U3, U4, V1, V2, V3, V4 

  DO KX = 2, IDIM4I, 2 
     DO KY = 2, IDIM4J, 2 
        K2 = ((KX - 2)*(KX - 2) + ASPECT2*(KY - 2)*(KY - 2))/4 
        IF (K2 == 0) CYCLE  
        TX = (KX - 2)/2./K2 
        TY = ASPECT*(KY - 2)/2./K2 
        U1 = TX*U(KX,KY-1) - TY*V(KX-1,KY) 
        U2 = (-TX*U(KX-1,KY-1)) - TY*V(KX,KY) 
        U3 = TX*U(KX,KY) + TY*V(KX-1,KY-1) 
        U4 = (-TX*U(KX-1,KY)) + TY*V(KX,KY-1) 
        V1 = TY*U(KX-1,KY) + TX*V(KX,KY-1) 
        V2 = TY*U(KX,KY) - TX*V(KX-1,KY-1) 
        V3 = (-TY*U(KX-1,KY-1)) + TX*V(KX,KY) 
        V4 = (-TY*U(KX,KY-1)) - TX*V(KX-1,KY) 
        U(KX-1,KY-1) = U1 
        V(KX-1,KY-1) = V1 
        U(KX,KY-1) = U2 
        V(KX,KY-1) = V2 
        U(KX-1,KY) = U3 
        V(KX-1,KY) = V3 
        U(KX,KY) = U4 
        V(KX,KY) = V4 
     END DO
  END DO
  RETURN  
END SUBROUTINE MCHGTINV


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  MDECOD
!
! FORMAL PARAMETERS:
!  - A  - reordered field 
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - rearranges the field from the ordering used in the REAL-REAL FFT
!    to the complex ordering
!______________________________________________________________________

SUBROUTINE MDECOD(A) 

  USE database

  IMPLICIT NONE

  COMPLEX(KIND=SIZE_OF_COMPLEX), INTENT(INOUT) :: A(IDIM2I,IDIM4J) 

  INTEGER :: KX, KY 
  COMPLEX(KIND=SIZE_OF_COMPLEX):: C, B, IMAG 

  IMAG = (0.,1.) 
  DO KY = 2, IDIM4J, 2 
     DO KX = 1, IDIM2I 
        B = A(KX,KY-1)
        C = A(KX,KY)
        A(KX,KY-1) = IMAG*C + B 
        A(KX,KY) = (-IMAG*C) + B 
     END DO
  END DO
  RETURN  
END SUBROUTINE MDECOD



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  MCOD
!
! FORMAL PARAMETERS:
!  - A  - reordered field 
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - rearranges the field from the complex ordering to the ordering 
!    used in the REAL-REAL FFT
!______________________________________________________________________

SUBROUTINE MCOD(A) 

  USE database

  IMPLICIT NONE

  COMPLEX(KIND=SIZE_OF_COMPLEX), INTENT(INOUT) :: A(IDIM2I,IDIM4J) 

  INTEGER :: KX, KY 
  COMPLEX(KIND=SIZE_OF_COMPLEX):: B, IMAG 

  IMAG = (0.,1.) 
  DO KX = 1, IDIM2I 
     DO KY = 2, IDIM4J, 2 
        B = (A(KX,KY)+A(KX,KY-1))/2. 
        A(KX,KY) = (A(KX,KY)-A(KX,KY-1))*IMAG/2. 
        A(KX,KY-1) = B 
     END DO
  END DO
  RETURN  
END SUBROUTINE MCOD


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  MODPRO
!
! FORMAL PARAMETERS:
!  - UA, VA, PA  - fields at instant N-1
!  - U, V, P     - fields at instant N
!  - MPRESS      - mean pressure
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - based on the fields at the time instant N, creates fields at the
!    instant N-1 while eliminating parasitic modes mof the leapfrog 
!
! NOTE:
!  On entry:  A      field ordering used in FFT REAL-REAL
!
!  On exit:   A      field in the complex ordering
!                              I.E.  FOURIER(KX,KY)=A(KX+1,KY+KY+1)
!                                    FOURIER(KX,-KY)=A(KX+1,KY+KY+2)
!______________________________________________________________________

SUBROUTINE MODPRO(UA, VA, PA, U, V, P, DT, P0, CO)

  USE parameters, ONLY : ASPECT, ASPECT2, VERBOSE
  USE database
  USE diagnostics, ONLY: OUTPUT1

  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL) , INTENT(IN) :: DT
  REAL(KIND=SIZE_OF_REAL) , INTENT(IN) :: P0
  REAL(KIND=SIZE_OF_REAL) , INTENT(IN) :: CO
  COMPLEX(KIND=SIZE_OF_COMPLEX) :: UA(IDIM2I,IDIM4J) 
  COMPLEX(KIND=SIZE_OF_COMPLEX) :: VA(IDIM2I,IDIM4J) 
  COMPLEX(KIND=SIZE_OF_COMPLEX) :: PA(IDIM2I,IDIM4J) 
  COMPLEX(KIND=SIZE_OF_COMPLEX) :: U(IDIM2I,IDIM4J) 
  COMPLEX(KIND=SIZE_OF_COMPLEX) :: V(IDIM2I,IDIM4J) 
  COMPLEX(KIND=SIZE_OF_COMPLEX) :: P(IDIM2I,IDIM4J) 

  INTEGER :: K2CUT, KX, KY
  REAL(KIND=SIZE_OF_REAL) :: K2 
  REAL(KIND=SIZE_OF_REAL) :: U1, V1, RAY2, RAY, RACO, IGO, TK, FSO, ANG 
  COMPLEX(KIND=SIZE_OF_COMPLEX) :: A0, A1, A2, ROT1, ROT2, IMAG
  COMPLEX(KIND=SIZE_OF_COMPLEX), DIMENSION(IDIM2I,IDIM4J) :: US, VS, PS

  IF ( VERBOSE > 1 )  WRITE(OUTPUT1,*)'  Eliminating parasitic leapfrog modes'     

  US(1:IDIM2I,1:IDIM4J) = U(1:IDIM2I,1:IDIM4J)
  VS(1:IDIM2I,1:IDIM4J) = V(1:IDIM2I,1:IDIM4J)
  PS(1:IDIM2I,1:IDIM4J) = P(1:IDIM2I,1:IDIM4J)

  K2CUT = ((IDIM3I/2) - 1)*((IDIM3I/2) - 1) 
  U1 = U(1,1) 
  V1 = V(1,1) 
  RAY2 = P0/(CO*CO) 
  RAY = SQRT(RAY2) 
  RACO = (P0/2)**.5 
  IMAG = CMPLX(0.,1.) 
  CALL MCHGT (U, V) 
  CALL MDECOD (U) 
  CALL MDECOD (V) 
  CALL MDECOD (P) 
  IGO = 20 
  DO KX = 1, IDIM2I 
     DO KY = 2, IDIM4J, 2 
10      CONTINUE  
        K2 = (KX - 1)*(KX - 1) + ASPECT2*(KY - 2)*(KY - 2)/4 
        IF (K2 > K2CUT) GO TO 20 
        IF (K2 == 0) GO TO 20 
        TK = SQRT(K2) 
        FSO = (1 + K2*RAY2)**(-.5) 
        A0 = (P(KX,KY-1)-CO*RAY2*V(KX,KY-1))*FSO 
        A1 = (TK*FSO/CO*P(KX,KY-1)+(IMAG*U(KX,KY-1)+FSO*V(KX,KY-1))/TK)*RACO
        A2 = (TK*FSO/CO*P(KX,KY-1)+((-IMAG*U(KX,KY-1))+FSO*V(KX,KY-1))/TK)*RACO 
        ANG = CO*DT/FSO 
        ROT1 = CMPLX(SQRT(1 - ANG*ANG),(-ANG)) 
        ROT2 = CMPLX(SQRT(1 - ANG*ANG),ANG) 
        PA(KX,KY-1) = A0*FSO + (A1*ROT1 + A2*ROT2)*TK*RACO*FSO/CO 
        UA(KX,KY-1) = ((-A1*ROT1) + A2*ROT2)*TK/RACO/2*IMAG 
        VA(KX,KY-1) = (-A0*K2*FSO/CO) + (A1*ROT1 + A2*ROT2)*TK/RACO/2*FSO
        A0 = (P(KX,KY)-CO*RAY2*V(KX,KY))*FSO 
        A1 = (TK*FSO/CO*P(KX,KY)+(IMAG*U(KX,KY)+FSO*V(KX,KY))/TK)*RACO 
        A2 = (TK*FSO/CO*P(KX,KY)+((-IMAG*U(KX,KY))+FSO*V(KX,KY))/TK)*RACO 
        PA(KX,KY) = A0*FSO + (A1*ROT1 + A2*ROT2)*TK*RACO*FSO/CO 
        UA(KX,KY) = ((-A1*ROT1) + A2*ROT2)*TK/RACO/2*IMAG 
        VA(KX,KY) = (-A0*K2*FSO/CO) + (A1*ROT1 + A2*ROT2)*TK/RACO/2*FSO 
        CYCLE  
20      CONTINUE 
        UA(KX,KY-1) = U(KX,KY-1) 
        VA(KX,KY-1) = V(KX,KY-1) 
        PA(KX,KY-1) = P(KX,KY-1) 
        UA(KX,KY) = U(KX,KY) 
        VA(KX,KY) = V(KX,KY) 
        PA(KX,KY) = P(KX,KY) 
     END DO
  END DO
  CALL MCOD (UA) 
  CALL MCOD (VA) 
  CALL MCOD (PA) 
  CALL MCHGTINV (UA, VA) 
  UA(1,1) = U1 
  VA(1,1) = V1 

  U(1:IDIM2I,1:IDIM4J) = US(1:IDIM2I,1:IDIM4J)
  V(1:IDIM2I,1:IDIM4J) = VS(1:IDIM2I,1:IDIM4J)
  P(1:IDIM2I,1:IDIM4J) = PS(1:IDIM2I,1:IDIM4J)

  RETURN  
END SUBROUTINE MODPRO


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  FMASK
!
! FORMAL PARAMETERS:
!  - NONE
!
! GLOBAL PARAMETERS:
!  - MASK    
!  - KCUTOFF   - cut-off wavenumber
!
! DESCRIPTION:
!  - creates the mask used in dealiasing
!______________________________________________________________________

SUBROUTINE FMASK 
  
  USE parameters, ONLY: ASPECT, ASPECT2, KCUTOFF, IDALIAS
  USE database
  IMPLICIT NONE

  INTEGER :: K, KMOD, H 

  DO H = 2, IDIM4J, 2 
     DO K = 2, IDIM4I, 2 
        KMOD = INT(SQRT((K - 2)*(K - 2) + ASPECT2*(H - 2)*(H - 2))*0.5 + &
             1.E-3) + 1 
! cutoff for 'square' dealiasing;
        IF ( (INT( (K-2)*0.5+1.E-3) + 1 >= KCUTOFF .OR. &
              INT( ASPECT*(H-2)*0.5+1.E-3) + 1 >= KCUTOFF) .AND. IDALIAS == 1 ) CYCLE  
! cutoff for 'circular' dealiasing;
        IF ( KMOD >= KCUTOFF .AND. IDALIAS == 2 ) CYCLE  
        MASK(K-1,H-1) = 1. 
        MASK(K-1,H) = 1. 
        MASK(K,H) = 1. 
        MASK(K,H-1) = 1. 
     END DO
  END DO
  RETURN  
END SUBROUTINE FMASK



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  FFTCR
!
! FORMAL PARAMETERS:
!  - A - transformed field
!
! GLOBAL PARAMETERS:
!  - JUMP1, JUMP2, INC1, INC2, IFAX, TRIGS, WORK
!
! DESCRIPTION:
!  - complex to real (inverse) Fourier transform
!______________________________________________________________________

SUBROUTINE FFTCR(A) 

  USE parameters
  USE database
  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL)  :: A(IDIM4I,IDIM4J) 

  INTEGER :: ISIGN, M 

  ISIGN = 1 
  M = IDIM3I - 1

! complex-complex transform in the direction OY
  A(2,1:IDIM4J) = A(IDIM3I,1:IDIM4J)

#if TURB2D_FFT_USED == SCILIB_CRAY
  CALL RFFTMLT (A, WORK, TRIGSJ, IFAXJ, INC2, JUMP2, IDIM3J, M, ISIGN) 
#elif TURB2D_FFT_USED == FFTW
  CALL RFFTWMLT (A, WORK, TRIGSJ, IFAXJ, INC2, JUMP2, IDIM3J, M, ISIGN)
#endif

! complex-real transform in the direction OX
  A(IDIM3I,:IDIM4J) = A(2,:IDIM4J) 
  A(2,:IDIM4J) = 0.
#if TURB2D_FFT_USED == SCILIB_CRAY
  CALL RFFTMLT (A, WORK, TRIGSI, IFAXI, INC1, JUMP1, IDIM3I, IDIM3J, ISIGN)
#elif TURB2D_FFT_USED == FFTW 
  CALL RFFTWMLT (A, WORK, TRIGSI, IFAXI, INC1, JUMP1, IDIM3I, IDIM3J, ISIGN)
#endif
  RETURN  
END SUBROUTINE FFTCR


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  FFTRC
!
! FORMAL PARAMETERS:
!  - A - transformed field
!
! GLOBAL PARAMETERS:
!  - JUMP1, JUMP2, INC1, INC2, IFAX, TRIGS, WORK
!
! DESCRIPTION:
!  - real to complex (direct) Fourier transform
!______________________________________________________________________

SUBROUTINE FFTRC(A) 

  USE parameters
  USE database

  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL)  :: A(IDIM4I,IDIM4J) 

  INTEGER :: ISIGN, M

  ISIGN = -1 
  M = IDIM3I - 1

! real-complex transform in the direction OX
#if TURB2D_FFT_USED == SCILIB_CRAY
  CALL RFFTMLT (A, WORK, TRIGSI, IFAXI, INC1, JUMP1, IDIM3I, IDIM3J, ISIGN)
#elif TURB2D_FFT_USED == FFTW 
  CALL RFFTWMLT (A, WORK, TRIGSI, IFAXI, INC1, JUMP1, IDIM3I, IDIM3J, ISIGN)
#endif
! the real part of the Nyquist frequency is put in the imaginary part 
! of the fundamental frequency
  A(2,:IDIM4J) = A(IDIM3I,:IDIM4J) 

! complex-complex transform in the direction OY
#if TURB2D_FFT_USED == SCILIB_CRAY
  CALL RFFTMLT (A, WORK, TRIGSJ, IFAXJ, INC2, JUMP2, IDIM3J, M, ISIGN) 
#elif TURB2D_FFT_USED == FFTW 
  CALL RFFTWMLT (A, WORK, TRIGSJ, IFAXJ, INC2, JUMP2, IDIM3J, M, ISIGN) 
#endif
  A(IDIM3I,:IDIM4J) = A(2,:IDIM4J) 
  A(2,1:IDIM4J) = 0. 

! Dealiasing - also removing the Nyquist frequency (COS), to ensure that
! upon differentiation this doesn't give an imaginary part (SIN)

  A(1:IDIM4I,1:IDIM4J) = A(1:IDIM4I,1:IDIM4J)*MASK(1:IDIM4I,1:IDIM4J) 

  RETURN  
END SUBROUTINE FFTRC



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  OROT
!
! FORMAL PARAMETERS:
!  - U, V - velocity components
!  - O    - vorticity
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - calculates the curl of the velocity field
!______________________________________________________________________

SUBROUTINE OROT(U, V, O)

  USE parameters
  USE database
  
  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL) , INTENT(IN) :: U(IDIM4I,IDIM4J) 
  REAL(KIND=SIZE_OF_REAL) , INTENT(IN) :: V(IDIM4I,IDIM4J) 
  REAL(KIND=SIZE_OF_REAL) , INTENT(OUT) :: O(IDIM4I,IDIM4J) 

  INTEGER :: K, H 


!  O=(DV/DX)-(DU/DY)
!  in spectral representation    O=I*K*V-I*H*U
!                                O(K-1,H-1)=(-K*V+H*U)/2
!                                O(K-1,H  )=(-K*V-H*U)/2
!                                O(K  ,H  )=(+K*V-H*U)/2
!                                O(K  ,H-1)=(+K*V+H*U)/2

!$OMP PARALLEL DO PRIVATE(K)
  DO H = 2, IDIM4J, 2 
     DO K = 2, IDIM4I, 2 
        O(K-1,H-1) = ((2 - K)*V(K,H-1)+ASPECT*(H-2)*U(K-1,H))*0.5 
        O(K,H-1) = ((K - 2)*V(K-1,H-1)+ASPECT*(H-2)*U(K,H))*0.5 
     END DO
     DO K = 2, IDIM4I, 2 
        O(K-1,H) = ((2 - K)*V(K,H)+ASPECT*(2-H)*U(K-1,H-1))*0.5 
        O(K,H) = ((K - 2)*V(K-1,H)+ASPECT*(2-H)*U(K,H-1))*0.5 
     END DO
  END DO
  RETURN  
END SUBROUTINE OROT



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  ODIV
!
! FORMAL PARAMETERS:
!  - U, V - velocity components
!  - O    - divergence
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - calculates the divergence of the velocity field
!______________________________________________________________________

SUBROUTINE ODIV(U, V, D)

  USE parameters
  USE database

  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL) , INTENT(IN) :: U(IDIM4I,IDIM4J) 
  REAL(KIND=SIZE_OF_REAL) , INTENT(IN) :: V(IDIM4I,IDIM4J) 
  REAL(KIND=SIZE_OF_REAL) , INTENT(OUT) :: D(IDIM4I,IDIM4J) 

  INTEGER :: K, H 


!  D=(DU/DX)+(DV/DY)
!  In spectral representation    D=I*K*U+I*H*V
!                                D(K-1,H-1)=(-K*U-H*V)/2
!                                D(K-1,H  )=(-K*U+H*V)/2
!                                D(K  ,H  )=(+K*U+H*V)/2
!                                D(K  ,H-1)=(+K*U-H*V)/2

!$OMP PARALLEL DO PRIVATE(K)
  DO H = 2, IDIM4J, 2 
     DO K = 2, IDIM4I, 2 
        D(K-1,H-1) = ((2 - K)*U(K,H-1)+ASPECT*(2-H)*V(K-1,H))*0.5 
        D(K,H-1) = ((K - 2)*U(K-1,H-1)+ASPECT*(2-H)*V(K,H))*0.5 
     END DO
     DO K = 2, IDIM4I, 2 
        D(K-1,H) = ((2 - K)*U(K,H)+ASPECT*(H-2)*V(K-1,H-1))*0.5 
        D(K,H) = ((K - 2)*U(K-1,H)+ASPECT*(H-2)*V(K,H-1))*0.5 
     END DO
  END DO
  RETURN  
END SUBROUTINE ODIV



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  OGRAD (Bartosz Protas, 07/01/2004)
!
! FORMAL PARAMETERS:
!  - U      -  the input field
!  - UX, UY -  components of the gradient field;
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - calculates the gradient of the input field;
!______________________________________________________________________

SUBROUTINE OGRAD(U, UX, UY)

  USE parameters
  USE database

  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL) , INTENT(IN)  :: U(IDIM4I,IDIM4J) 
  REAL(KIND=SIZE_OF_REAL) , INTENT(OUT) :: UX(IDIM4I,IDIM4J) 
  REAL(KIND=SIZE_OF_REAL) , INTENT(OUT) :: UY(IDIM4I,IDIM4J) 

  INTEGER :: K, H 

!  In spectral representation    UX=I*K*U
!                                UY=I*H*V
!                                UX(K-1,H-1)= -K*U/2,   UY(K-1,H-1)= -H*U/2
!                                UX(K-1,H  )= -K*U/2,   UY(K-1,H  )= +H*U/2
!                                UX(K  ,H  )= +K*U/2,   UY(K  ,H  )= +H*U/2
!                                UX(K  ,H-1)= +K*U/2,   UY(K  ,H-1)= -H*U/2
  DO H = 2, IDIM4J, 2 
     DO K = 2, IDIM4I, 2 
        UX(K-1,H-1) = 0.5 * (2 - K) * U(K  ,H-1)
        UX(K  ,H-1) = 0.5 * (K - 2) * U(K-1,H-1)
        UY(K-1,H-1) = 0.5 * (2 - H) * U(K-1,H  ) * ASPECT
        UY(K  ,H-1) = 0.5 * (2 - H) * U(K  ,H  ) * ASPECT
     END DO
     DO K = 2, IDIM4I, 2 
        UX(K-1,H  ) = 0.5 * (2 - K) * U(K  ,H  )
        UX(K  ,H  ) = 0.5 * (K - 2) * U(K-1,H  )
        UY(K-1,H  ) = 0.5 * (H - 2) * U(K-1,H-1) * ASPECT
        UY(K  ,H  ) = 0.5 * (H - 2) * U(K  ,H-1) * ASPECT
     END DO
  END DO

END SUBROUTINE OGRAD


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  OLAPDIR
!
! FORMAL PARAMETERS:
!  - F  - input field
!  - L  - laplacian of the input field
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - calculates the laplacian of a field
!______________________________________________________________________

SUBROUTINE OLAPDIR(F, L)

  USE parameters
  USE database

  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL) , INTENT(IN) :: F(IDIM4I,IDIM4J) 
  REAL(KIND=SIZE_OF_REAL) , INTENT(OUT) :: L(IDIM4I,IDIM4J) 

  INTEGER :: K, H 
  REAL(KIND=SIZE_OF_REAL) :: K2

!  L=(D2F/DX2)+(D2F/DY2) in spectral representation L=-K2*F
  DO H = 1, IDIM2J
     DO K = 1, IDIM2I 
        K2 = (K - 1)*(K - 1) + ASPECT2*(H - 1)*(H - 1) 
        L(2*K-1,2*H-1) = -K2*F(2*K-1,2*H-1) 
        L(2*K,2*H-1) = -K2*F(2*K,2*H-1) 
     END DO
     DO K = 1, IDIM2I 
        K2 = (K - 1)*(K - 1) + ASPECT2*(H - 1)*(H - 1) 
        L(2*K-1,2*H) = -K2*F(2*K-1,2*H) 
        L(2*K,2*H) = -K2*F(2*K,2*H) 
     END DO
  END DO
  RETURN  
END SUBROUTINE OLAPDIR


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  OLAPINV
!
! FORMAL PARAMETERS:
!  - L  - input field
!  - F  - laplacian of the input field
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - calculates the inverse laplacian of a field (i.e. solves 
!    the Laplace equation)
!______________________________________________________________________

SUBROUTINE OLAPINV(L, F)

  USE parameters
  USE database

  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL) , INTENT(IN) :: L(IDIM4I,IDIM4J) 
  REAL(KIND=SIZE_OF_REAL) , INTENT(OUT) :: F(IDIM4I,IDIM4J) 

  INTEGER :: K, H 
  REAL(KIND=SIZE_OF_REAL) :: K2


!  ((D2F/DX2)+(D2F/DY2))=L in spectral representation F=-L/K2
!$OMP PARALLEL DO PRIVATE(K2)
  DO H = 2, IDIM2J 
     K2 = 1./(ASPECT2*(H - 1)*(H - 1)) 
     F(1,2*H-1) = -L(1,2*H-1)*K2 
     F(1,2*H) = -L(1,2*H)*K2 
     F(2,2*H) = -L(2,2*H)*K2 
     F(2,2*H-1) = -L(2,2*H-1)*K2 
  END DO
  
!$OMP PARALLEL DO PRIVATE(K2,H)
  DO K = 2, IDIM2I 
     DO H = 1, IDIM2J 
        K2 = 1./((K - 1)*(K - 1) + ASPECT2*(H - 1)*(H - 1)) 
        F(2*K-1,2*H-1) = -L(2*K-1,2*H-1)*K2 
        F(2*K-1,2*H) = -L(2*K-1,2*H)*K2 
        F(2*K,2*H) = -L(2*K,2*H)*K2 
        F(2*K,2*H-1) = -L(2*K,2*H-1)*K2
     END DO
  END DO
  
  RETURN  
END SUBROUTINE OLAPINV



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  OVELOCITY
!
! FORMAL PARAMETERS:
!  - O    - vorticty field
!  - U, V - velocity components
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - based on the vorticity field, calculates the velocity components
!    (i.e., inverts the curl operator)
!______________________________________________________________________

SUBROUTINE OVELOCITY(O, U, V)

  USE parameters
  USE database

  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL)  :: O(IDIM4I,IDIM4J) 
  REAL(KIND=SIZE_OF_REAL) , INTENT(OUT) :: U(IDIM4I,IDIM4J) 
  REAL(KIND=SIZE_OF_REAL) , INTENT(OUT) :: V(IDIM4I,IDIM4J) 

  INTEGER :: K, H 
  REAL(KIND=SIZE_OF_REAL), DIMENSION(IDIM4I,IDIM4J) :: F 

! streamfunction based on vorticity (meteo convention "-")
  CALL OLAPINV (O, F) 

! Velocity components from streamfunction
!  U ROT=-(DF/DY) in spectral representation U ROT=-I*H*F
!$OMP PARALLEL DO PRIVATE(H)
  DO K = 1, IDIM4I 
!DIR$ IVDEP
     DO H = 2, IDIM4J, 2 
        U(K,H-1) = ASPECT*(H - 2)*0.5*F(K,H) 
        U(K,H) = ASPECT*(2 - H)*0.5*F(K,H-1) 
     END DO
  END DO

!  V ROT=+(DF/DX) in spectral representation V ROT=+I*K*F
!$OMP PARALLEL DO PRIVATE(K)
  DO H = 1, IDIM4J 
!DIR$ IVDEP
     DO K = 2, IDIM4I, 2 
        V(K-1,H) = (2 - K)*0.5*F(K,H) 
        V(K,H) = (K - 2)*0.5*F(K-1,H) 
     END DO
  END DO

  RETURN  
END SUBROUTINE OVELOCITY


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  SAVE_PHYSICAL_FIELDS (Bartek Protas, 18/12/2003)
!
! PARAMETERS:
!  - N - time step
!
! DESCRIPTION:
!  - saves physical fields in the binary form
!  - saves physical fields in the video (MPG) format
!______________________________________________________________________

SUBROUTINE SAVE_PHYSICAL_FIELDS( N )

  USE database
  USE parameters
  USE diagnostics
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: N

! vorticity (physical space)
#ifdef MPEG            
  IF ((ISVORT==1 .AND. MOD(N-1,NSVORT)==0).OR.(IMPGVORT==1 .AND. MOD(N-1,NMPGVORT)==0)) THEN
#else
  IF (ISVORT==1 .AND. MOD(N-1,NSVORT)==0) THEN
#endif     

    WORK(1:IDIM4I,1:IDIM4J,1) = PO(1:IDIM4I,1:IDIM4J) - CORIOL 

  END IF
  IF (ISVORT==1 .AND. MOD(N-1,NSVORT)==0)       WRITE (NFVORT) (WORK(1:IDIM3I,1:IDIM3J,1))
#ifdef MPEG        
  IF (IMPGVORT==1 .AND. MOD(N-1,NMPGVORT)==0)   CALL AddFrameToMovie(MPGVORT,(WORK(1:IDIM3I,1:IDIM3J,1)))
#endif

! pressure (physical space)
  IF (ISPRESS==1 .AND. MOD(N-1,NSPRESS)==0)     WRITE (NFPRESS) (PP(1:IDIM3I,1:IDIM3J))
#ifdef MPEG  
  IF (IMPGPRESS==1 .AND. MOD(N-1,NMPGPRESS)==0) CALL AddFrameToMovie(MPGPRESS,(PP(1:IDIM3I,1:IDIM3J)))
#endif

! divergence (physical space)
  IF (ISDIV==1 .AND. MOD(N-1,NSDIV)==0)         WRITE (NFDIV) (PD(1:IDIM3I,1:IDIM3J))
#ifdef MPEG
  IF (IMPGDIV==1 .AND. MOD(N-1,NMPGDIV)==0)     CALL AddFrameToMovie(MPGDIV,(PD(1:IDIM3I,1:IDIM3J)))
#endif

! potential vorticity (physical space)
#ifdef MPEG
  IF ((ISVPOT==1 .AND. MOD(N-1,NSVPOT)==0).OR.(IMPGVPOT==1 .AND. MOD(N-1,NMPGVPOT)==0)) THEN
#else
  IF (ISVPOT==1 .AND. MOD(N-1,NSVPOT)==0) THEN
#endif

    WORK(1:IDIM4I,1:IDIM4J,1) = PO(1:IDIM4I,1:IDIM4J)/(MEAN_PRESS + PP(1:IDIM4I,1:IDIM4J)) - CORIOL/MEAN_PRESS
  END IF
     
  IF (ISVPOT==1 .AND. MOD(N-1,NSVPOT)==0)       WRITE (NFVPOT) (WORK(1:IDIM3I,1:IDIM3J,1))
#ifdef MPEG
  IF (IMPGVPOT==1 .AND. MOD(N-1,NMPGVPOT)==0)   CALL AddFrameToMovie(MPGVPOT,(WORK(1:IDIM3I,1:IDIM3J,1)))
#endif

! streamfunction (physical space)
  IF (ISSF==1 .AND. MOD(N-1,NSSF)==0)       WRITE (NFSF) (PF(1:IDIM3I,1:IDIM3J))
#ifdef MPEG
  IF (IMPGSF==1 .AND. MOD(N-1,NMPGSF)==0)   CALL AddFrameToMovie(MPGSF,(PF(1:IDIM3I,1:IDIM3J)))
#endif

! horizontal velocity (physical space)
  IF (ISU==1 .AND. MOD(N-1,NSU)==0)             WRITE (NFU) (PU(1:IDIM3I,1:IDIM3J))
#ifdef MPEG
  IF (IMPGU==1 .AND. MOD(N-1,NMPGU)==0)         CALL AddFrameToMovie(MPGU,(PU(1:IDIM3I,1:IDIM3J)))
#endif        

! vertical velocity (physical space)
  IF (ISV==1 .AND. MOD(N-1,NSV)==0)             WRITE (NFV) (PV(1:IDIM3I,1:IDIM3J))
#ifdef MPEG
  IF (IMPGV==1 .AND. MOD(N-1,NMPGV)==0)         CALL AddFrameToMovie(MPGV,(PV(1:IDIM3I,1:IDIM3J)))
#endif

! velocity modulus (physical space)
#ifdef MPEG
  IF ((ISMODV==1 .AND. MOD(N-1,NSMODV)==0).OR.(IMPGMODV==1 .AND. MOD(N-1,NMPGMODV)==0)) THEN
#else
  IF (ISMODV==1 .AND. MOD(N-1,NSMODV)==0) THEN
#endif

    WORK(1:IDIM4I,1:IDIM4J,1) = SQRT(PU(1:IDIM4I,1:IDIM4J)**2+PV(1:IDIM4I,1:IDIM4J)**2)

  END IF

  IF (ISMODV==1 .AND. MOD(N-1,NSMODV)==0)       WRITE (NFMODV) (WORK(1:IDIM3I,1:IDIM3J,1))
#ifdef MPEG
  IF (IMPGMODV==1 .AND. MOD(N-1,NMPGMODV)==0)   CALL AddFrameToMovie(MPGMODV,(WORK(1:IDIM3I,1:IDIM3J,1)))
#endif
	
! Bernoulli function: PB = PP+(PU**2+PV**2)*0.5 (physical space)
#ifdef MPEG
  IF ((ISBERN==1 .AND. MOD(N-1,NSBERN)==0).OR.(IMPGBERN==1 .AND. MOD(N-1,NMPGBERN)==0)) THEN
#else
  IF (ISBERN==1 .AND. MOD(N-1,NSBERN)==0) THEN
#endif

    WORK(1:IDIM4I,1:IDIM4J,1) = PP(1:IDIM4I,1:IDIM4J) + (PU(1:IDIM4I,1:IDIM4J)**2+PV(1:IDIM4I,1:IDIM4J)**2)*0.5

  END IF

  IF (ISBERN==1 .AND. MOD(N-1,NSBERN)==0)       WRITE (NFBERN) (WORK(1:IDIM3I,1:IDIM3J,1))
#ifdef MPEG
  IF (IMPGBERN==1 .AND. MOD(N-1,NMPGBERN)==0)   CALL AddFrameToMovie(MPGBERN,(WORK(1:IDIM3I,1:IDIM3J,1)))
#endif

! pressure slice (physical space)
  IF (ISCUTP==1 .AND. MOD(N-1,NSCUT)==0)      WRITE (NFCUTP) PP(1:IDIM3I,NUMCUT)

! vertical slice of v_x (physical space)
  IF (ISCUTV==1 .AND. MOD(N-1,NSCUT)==0)      WRITE (NFCUTV) PV(1:IDIM3I,NUMCUT)

! vertical slice of v_y (physical space)
  IF (ISCUTU==1 .AND. MOD(N-1,NSCUT)==0)      WRITE (NFCUTU) PU(1:IDIM3I,NUMCUT)
         
END SUBROUTINE SAVE_PHYSICAL_FIELDS


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  SAVE_SPECTRAL_FIELDS (Bartek Protas, 13/01/2004)
!
! PARAMETERS:
!  - N - time step
!
! DESCRIPTION:
!  - saves spectral fields in the binary form
!______________________________________________________________________

SUBROUTINE SAVE_SPECTRAL_FIELDS( N )

  USE database
  USE parameters
  USE diagnostics
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: N
  INTEGER :: K, H
  
	
  IF ( VERBOSE > 1 ) WRITE(OUTPUT1,*) '  Saving spectral field'
  IF ( N /= NM ) THEN
     WRITE(UNIT=NAME, FMT='("field_",I8.8)') NSTEP
  ELSE
     WRITE(UNIT=NAME, FMT='("field_final")')
  ENDIF
  OPEN(STATUS="REPLACE", UNIT=NVM1, FILE=NAME, FORM="UNFORMATTED")          
  WRITE (NVM1) NSTEP, TIMEA, TIME
  WRITE (NVM1) ((PUA(K,H), K=1,IDIM4I), H=1,IDIM4J)
  WRITE (NVM1) ((PVA(K,H), K=1,IDIM4I), H=1,IDIM4J)
  WRITE (NVM1) ((PPA(K,H), K=1,IDIM4I), H=1,IDIM4J)
  WRITE (NVM1) ((PU(K,H), K=1,IDIM4I), H=1,IDIM4J)
  WRITE (NVM1) ((PV(K,H), K=1,IDIM4I), H=1,IDIM4J)
  WRITE (NVM1) ((PP(K,H), K=1,IDIM4I), H=1,IDIM4J)
  IF (PENALIZATION /=0) WRITE (NVM1) ((KSI(K,H), K=1,IDIM4I), H=1, IDIM4J)
  IF (SPONGE /= 0) WRITE (NVM1) ((SPONGE_MASK(K,H), K=1,IDIM4I), H=1,IDIM4J)
  IF (WALLS /= 0) WRITE (NVM1) ((WALL_MASK(K,H), K=1,IDIM4I), H=1,IDIM4J)
  CLOSE(NVM1)     

END SUBROUTINE SAVE_SPECTRAL_FIELDS

!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  CALCULATE_DIAGNOSTICS (Bartek Protas, 18/12/2003)
!
! PARAMETERS:
!  - N - time step
!
! DESCRIPTION:
!  - calculates various diagnostics; the data must be in physical 
!    space representation;
!______________________________________________________________________

SUBROUTINE CALCULATE_DIAGNOSTICS( N )

  USE database
  USE parameters
  USE diagnostics
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: N

  INTEGER :: K, N0
  REAL(KIND=SIZE_OF_REAL) :: DE, DZP
  REAL(KIND=SIZE_OF_REAL), SAVE :: E1, E2, ZP1, ZP2
  REAL(KIND=SIZE_OF_REAL), EXTERNAL :: VNORML2

  IF ( N == 0 ) NDIAG = 0

!!/////////////////////////////////////////////////////////////////////////
!!///// Update made by Bartek Protas on the 27th June  2004 ///////////////
! This should be fixed ...
  MACH = 1.0
  RO = 1.0
!!/////////////////////////////////////////////////////////////////////////

                                                                                
! Saving diagnostics to a file;
  IF ( NDIAG == NDIAG_MAX ) THEN
! This was computed at the previous time step, so things need to be shifted back by 1;
     N0 = N - NDIAG_MAX * NINV - 1
     IF ( INCOMP == 0 ) THEN
        DO K=1,NDIAG_MAX
           WRITE(NFDIAG,8051) (NINIT+N0+K*NINV)*DT, DDIAG(K,1:20)
        ENDDO
     ELSE
        DO K=1,NDIAG_MAX
           WRITE(NFDIAG,8052) (NINIT+N0+K*NINV)*DT, DDIAG(K,1), DDIAG(K,5), DDIAG(K,8:18)
        ENDDO
     ENDIF
     NDIAG = 0
  ENDIF                                                                               
  
! Calculating invariants and maximum fluctuations
  IF ( MOD(N,NINV)==0 ) THEN 
     CALL DINVAR (E1, ZP1) 
     SEPARLF = 0.0 
     CALL DFLUCT 

! Calculating vorticity gradient and its norm;
     CALL OGRAD( PO, WORK(1:IDIM4I,1:IDIM4J,1), WORK(1:IDIM4I,1:IDIM4J,2) )
     GRAD_VOR_L2 = VNORML2( WORK(1:IDIM4I,1:IDIM4J,1), WORK(1:IDIM4I,1:IDIM4J,2) ) 

     NDIAG = NDIAG + 1
! Storing the diagnostics
     DDIAG(NDIAG,1)  = ETOT
     DDIAG(NDIAG,2)  = EKIN
     DDIAG(NDIAG,3)  = EPOT
     DDIAG(NDIAG,4)  = ELIN
     DDIAG(NDIAG,5)  = ZTOT
     DDIAG(NDIAG,6)  = ZPOT
     DDIAG(NDIAG,7)  = ZLIN 

     DDIAG(NDIAG,8)  = GRAD_VOR_L2
     DDIAG(NDIAG,9)  = DIV_L1
         
     DDIAG(NDIAG,10) = V0MAX
     DDIAG(NDIAG,11) = UMIN
     DDIAG(NDIAG,12) = UMAX
     DDIAG(NDIAG,13) = VMIN
     DDIAG(NDIAG,14) = VMAX
     DDIAG(NDIAG,15) = PMIN
     DDIAG(NDIAG,16) = PMAX

     DDIAG(NDIAG,17) = PI*DT*KMAX*V0MAX     ! CFL
     DDIAG(NDIAG,18) = NU
     DDIAG(NDIAG,19) = MACH
     DDIAG(NDIAG,20) = RO     

! Writing data to the screen;
     IF ( VERBOSE > 0 ) THEN
        IF ( INCOMP == 0 ) THEN    
           IF (N == 0 ) WRITE(OUTPUT1,8043) 'Time', 'Total energy', 'Kinetic energy','Potential energy', &
                                            'Linear energy','Total enstrophy','Potential enstrophy',&
                                            'Linear enstrophy','Grad Vor','Div','Vel_Max', 'Umin', 'Umax', &
                                            'Vmin',  'Vmax', 'Pmin', 'Pmax', 'CFL', 'nu', 'Mach number', 'Rossby number'
           WRITE(OUTPUT1,8041) (NINIT+N)*DT, DDIAG(NDIAG,1:20)
        ELSE
           IF ( N == 0 ) WRITE(OUTPUT1,8040) 'Time', 'Energy', 'Enstrophy', 'Grad Vor',   &
                                             'Div', 'Vel_Max', 'Umin', 'Umax', 'Vmin',    &
                                             'Vmax', 'Pmin', 'Pmax', 'CFL', 'nu'
           WRITE(OUTPUT1,8042) (NINIT+N)*DT, DDIAG(NDIAG,1), DDIAG(NDIAG,5), DDIAG(NDIAG,8:18)   
        ENDIF
     ENDIF
     
  ELSE IF ( MOD(N,NINV)==1 ) THEN 
     CALL DINVAR (E2, ZP2) 
     DE      = (E2  - E1 )*100.0 / E1 
     DZP     = (ZP2 - ZP1)*100.0 / ZP1 
     IF ( N > 1 ) SEPARLF = MAX(ABS(DE),ABS(DZP)) 
  ENDIF
  
8040 FORMAT((5X,5(A14)),(7X,7(A11)),(1X,A14),(1X,A11))
8041 FORMAT(21X,21(1PE14.6))
8042 FORMAT((5X,5(1PE14.6)),(7X,7(1PE11.3)),(1X,1PE14.6),(1X,1PE11.3)) 
8043 FORMAT(21X,21(A20))
8051 FORMAT(21X,21(1PE14.6))
8052 FORMAT(14X,14(1PE14.6))

END SUBROUTINE CALCULATE_DIAGNOSTICS


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Subroutine  PRESSURE_CORRECTION
!
! FORMAL PARAMETERS:
!  - N - time step;
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - enforces incompressibility of the velocity field using the 
!    "pressure corrrrection" appraoch
!  - calculates actual pressure (from the Poisson equation)
!______________________________________________________________________

SUBROUTINE PRESSURE_CORRECTION( N )

  USE parameters
  USE database
  USE diagnostics, ONLY: DIV_L1

  IMPLICIT NONE

  INTEGER, INTENT(IN) :: N
  INTEGER :: H, K
  REAL(KIND=SIZE_OF_REAL) ::  DIV1, DIV2
  REAL(KIND=SIZE_OF_REAL) , EXTERNAL :: SNORML1
!
! first, calculating the divergence of the intermediate velocity field;
  CALL ODIV (PU, PV, PD) 
   
! TEST - calculating the divergance before pressure correction;
  IF ( VERBOSE > 2 )  DIV1 = SNORML1( PD )

! now solving the Poisson equation for the pressure correction 
! (recycling PF);
  CALL OLAPINV (PD, PF)      
! Now subtracting the pressure correction to enforce incompressibility;
! the horizontal component;
!CDIR NODEP
!$OMP PARALLEL DO PRIVATE(K)
  DO H = 2, IDIM4J, 2 
!CDIR NODEP
     DO K = 2, IDIM4I, 2 
        PU(K-1,H-1) = PU(K-1,H-1) + (K - 2)*0.5*PF(K  ,H-1)               
        PU(K  ,H-1) = PU(K  ,H-1) + (2 - K)*0.5*PF(K-1,H-1) 
     END DO
!CDIR NODEP         
     DO K = 2, IDIM4I, 2
        PU(K-1,H) = PU(K-1,H) + (K - 2)*0.5*PF(K  ,H)
        PU(K  ,H) = PU(K  ,H) + (2 - K)*0.5*PF(K-1,H)
     END DO
  END DO
      
! the vertical component;
!CDIR NODEP
!$OMP PARALLEL DO PRIVATE(K)
  DO H = 2, IDIM4J, 2
!CDIR NODEP
     DO K = 2, IDIM4I, 2
        PV(K-1,H-1) = PV(K-1,H-1) + ASPECT*(H - 2)*0.5*PF(K-1,H) 
        PV(K  ,H-1) = PV(K  ,H-1) + ASPECT*(H - 2)*0.5*PF(K  ,H) 
     END DO
!CDIR NODEP            
     DO K = 2, IDIM4I, 2
        PV(K-1,H)   = PV(K-1,H)   + ASPECT*(2 - H)*0.5*PF(K-1,H-1) 
        PV(K  ,H)   = PV(K  ,H)   + ASPECT*(2 - H)*0.5*PF(K  ,H-1) 
     END DO
  END DO


! Calculating the divergance after pressure correction;
  IF ( MOD(N,NINV)==(NINV-1) .OR. VERBOSE > 2 ) THEN
     CALL ODIV (PU, PV, PD) 
     DIV2 = SNORML1( PD )
     DIV_L1 = DIV2

! saving integrated divergence;
     IF ( VERBOSE > 2 ) THEN
        OPEN(STATUS="UNKNOWN", UNIT=99, FILE="press_corr.dat", FORM="FORMATTED", POSITION="APPEND")
        WRITE(99,*) DIV1, DIV2
        CLOSE(99)
     END IF
  ENDIF

! Now assembling the RHS for the Poisson pressure solver;
! Calculating the term (du/dx);
!CDIR NODEP
!$OMP PARALLEL DO PRIVATE(K)
  DO H = 2, IDIM4J, 2 
!CDIR NODEP
     DO K = 2, IDIM4I, 2 
        PD(K-1,H-1) = (K - 2)*0.5*PU(K  ,H-1)               
        PD(K  ,H-1) = (2 - K)*0.5*PU(K-1,H-1) 
     END DO
!CDIR NODEP         
     DO K = 2, IDIM4I, 2
        PD(K-1,H)   = (K - 2)*0.5*PU(K  ,H)
        PD(K  ,H)   = (2 - K)*0.5*PU(K-1,H)
     END DO
  END DO

! calculating the product (du/dx)^2 in physical space;
!CDIR IEXPAND
  CALL FFTCR (PD) 

!$OMP PARALLEL DO
  DO H=1,IDIM4J
     PO(1:IDIM4I,H) = PD(1:IDIM4I,H) * PD(1:IDIM4I,H)
  ENDDO

! Calculating the terms (du/dy) and (dv/dx)
!CDIR NODEP
!$OMP PARALLEL DO PRIVATE(K)
  DO H = 2, IDIM4J, 2 
!CDIR NODEP
     DO K = 2, IDIM4I, 2 
        PD(K-1,H-1) = (K - 2)*0.5*PV(K  ,H-1)               
        PD(K  ,H-1) = (2 - K)*0.5*PV(K-1,H-1) 
     END DO
!CDIR NODEP         
     DO K = 2, IDIM4I, 2
        PD(K-1,H)   = (K - 2)*0.5*PV(K  ,H)
        PD(K  ,H)   = (2 - K)*0.5*PV(K-1,H)
     END DO
  END DO

!CDIR NODEP
!$OMP PARALLEL DO PRIVATE(K)
  DO H = 2, IDIM4J, 2
!CDIR NODEP
     DO K = 2, IDIM4I, 2
        PF(K-1,H-1) = ASPECT*(H - 2)*0.5*PU(K-1,H) 
        PF(K  ,H-1) = ASPECT*(H - 2)*0.5*PU(K  ,H) 
     END DO
!CDIR NODEP            
     DO K = 2, IDIM4I, 2
        PF(K-1,H)   = ASPECT*(2 - H)*0.5*PU(K-1,H-1) 
        PF(K  ,H)   = ASPECT*(2 - H)*0.5*PU(K  ,H-1) 
     END DO
  END DO

! Calculating the product (du/dy)(dv/dx) in physical space;
!CDIR IEXPAND
  CALL FFTCR (PD) 
!CDIR IEXPAND
  CALL FFTCR (PF) 

!$OMP PARALLEL DO
  DO H=1,IDIM4J
     PO(1:IDIM4I,H) = - 2.0 * (PO(1:IDIM4I,H) + PD(1:IDIM4I,H) * PF(1:IDIM4I,H))
  ENDDO

! Transforming the RHS back to Fourier space;
  CALL FFTRC (PO) 

! Solving the Poisson equation for pressure PP;
  CALL OLAPINV (PO, PP)      

! Ensuring that pressure has a zero mean mode;
  PP(1,1) = 0.0

END SUBROUTINE PRESSURE_CORRECTION


!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Function  SNORML1
!
! FORMAL PARAMETERS:
!  - U - the scalar field
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - calculates the L1 norm of a scalar field
!______________________________________________________________________

FUNCTION SNORML1(U)

  USE parameters
  USE database

  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL), INTENT(INOUT) ::  U(IDIM4I,IDIM4J)

  REAL(KIND=SIZE_OF_REAL) :: SNORML1

! first, going to the real space;
  CALL FFTCR (U) 
      
! Now summing the field (in reduced units);
  SNORML1 = SUM( ABS(U(1:IDIM4I,1:IDIM4J)) ) * DX * DX * ASPECT 

! bringing the divergence field back to Fourier space;
  CALL FFTRC (U) 

END FUNCTION SNORML1



!______________________________________________________________________
!______________________________________________________________________
!______________________________________________________________________
!
!    $$ Function  VNORML2 (Bartosz Protas, 07/01/2004)
!
! FORMAL PARAMETERS:
!  - U, V - components of the vector field
!
! GLOBAL PARAMETERS:
!
! DESCRIPTION:
!  - calculates the L2 norm of the vector field using the Parceval identity;
!______________________________________________________________________

FUNCTION VNORML2(U, V)

  USE parameters
  USE database

  IMPLICIT NONE

  REAL(KIND=SIZE_OF_REAL), INTENT(IN) ::  U(IDIM4I,IDIM4J)
  REAL(KIND=SIZE_OF_REAL), INTENT(IN) ::  V(IDIM4I,IDIM4J)

  REAL(KIND=SIZE_OF_REAL) :: VNORML2

! Computing the norm in Fourier space;
  VNORML2 = SQRT(SUM( U(1:IDIM4I,1:IDIM4J)**2 + V(1:IDIM4I,1:IDIM4J)**2))

END FUNCTION VNORML2

