/*
This file is part of a collection of files written by Alexandre
AZZALINI, PhD Student at the French Laboratoire de Meteorologie
Dynamique du CNRS, Ecole Normale Superieure, PARIS, FRANCE.
Copyright Alexandre AZZALINI, CNRS-2001
This is free software. You can redistribute it and/or modify it as
soon as you do not remove this comment.
This piece of program comes without any warranty;without	
even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.	 
 */
#include <stdlib.h>
#include <colormap.h>
#include <colortype.h>
#include <math.h>

typedef struct
{
  int nx;
  int ny;
  int saut;
  color* image;
} image;

static image NULL_image=
{
  0,
  0,
  0,
  NULL,
};

#define ADRESSE(IMAGE,x,y) (IMAGE).image + ((x) + (y) * ((IMAGE).saut))
#define ELEMENT(IMAGE,x,y) (IMAGE).image[(x) + (y) * ((IMAGE).saut)]
#define isnull(IMAGE) ( ( (IMAGE).nx == 0 ) || ( (IMAGE).ny == 0 ) || ( (IMAGE).saut == 0 ) || ( (IMAGE).image == NULL ) )

image create_image(int nx,int ny, color bg)
{
  image result;
  register int i;
  register color* ptr;

  if (nx<=0 || ny <=0)
    return (NULL_image);

  result.nx=nx;
  result.ny=ny;
  result.saut=nx;
  if ((ptr = (result.image = (color*) malloc(nx*ny*sizeof(color))))==NULL)
    return (NULL_image);
  
  for (i=0;i<nx*ny;i++,ptr++)
    *ptr = bg;

  return(result);
}

image sub_image(image mother,int origin_x, int origin_y,int nx,int ny)
{
  image result;

  if (nx<=0 || ny<=0 || origin_x<0 || origin_y<0)
    return (NULL_image);

  if (((origin_x+nx)>mother.nx)||((origin_y+ny)>mother.ny))
    return (NULL_image);

  result.nx=nx;
  result.ny=ny;
  result.saut=mother.saut;
  result.image = ADRESSE(mother,origin_x,origin_y);

  return (result);
}

int copy_image(image source, image dest)
{
  register int i,j,nx,ny;

  if (isnull(source) || isnull(dest) )
    return (EXIT_FAILURE);

  if (source.nx != dest.nx)
    return (EXIT_FAILURE);

  if (source.ny != dest.ny)
    return (EXIT_FAILURE);

  nx=source.nx;
  ny=source.ny;

  for (j=0;j<ny;j++)
    for (i=0;i<nx;i++)      
      ELEMENT(dest,i,j)=ELEMENT(source,i,j);

  return(EXIT_SUCCESS);
}


int turn_image_LR(image source, image dest, int left)
{
  register int i,j,nx,ny;

  if (isnull(source) || isnull(dest) )
    return (EXIT_FAILURE);

  if (source.nx != dest.ny)
    return (EXIT_FAILURE);

  if (source.ny != dest.nx)
    return (EXIT_FAILURE);

  nx=source.nx;
  ny=source.ny;

  if (left==1)
    {
      for (j=0;j<ny;j++)
	for (i=0;i<nx;i++)      
	  ELEMENT(dest,j,nx-i-1)=ELEMENT(source,i,j);
    }
  
  if (left==-1)
    {
      for (j=0;j<ny;j++)
	for (i=0;i<nx;i++)      
	  ELEMENT(dest,ny-j-1,i)=ELEMENT(source,i,j);
    }

  if (left==0)
    return(EXIT_FAILURE);

  return(EXIT_SUCCESS);
}

int fill_image(image im,color col)
{
  register int i,j,nx,ny;

  nx=im.nx;
  ny=im.ny;

  if (isnull(im))
    return (EXIT_FAILURE);

  for (j=0;j<ny;j++)
    for (i=0;i<nx;i++)      
      ELEMENT(im,i,j)=col;
  
  return(EXIT_SUCCESS);
}

int set_pixel(image im,int nx, int ny,color col)
{
  if (nx>=im.nx || nx<0 || ny>=im.ny || ny<0)
    return (EXIT_FAILURE);

  ELEMENT(im,nx,ny)=col;
  return(EXIT_SUCCESS);
}

int draw_line(image im,int inx, int iny, int fnx, int fny, color col)
{
  register int i,j,k;
  register float coefdir;

  
  if (inx>fnx)
    {
      i=fnx,fnx=inx,inx=i;
      i=fny,fny=iny,iny=i;
    }

  if (inx>=im.nx || inx<0 || iny>=im.ny || iny<0 || fnx>=im.nx || fnx<0 || fny>=im.ny || fny<0)
    return (EXIT_FAILURE);
  
  if (inx==fnx)
    {
      if (iny>fny)
	i=fny,fny=iny,iny=i;
      for (j=iny;j<=fny;j++)
	ELEMENT(im,inx,j)=col;
    }
  else
    {
      coefdir = ((float) (fny-iny))/(fnx-inx);
      if (iny<=fny)
	for (i=inx;i<fnx;i++)
	  {
	    k = (int) floor(iny+coefdir*(i-inx+0.5));
	    for (j=(int) floor(iny+coefdir*(i-inx)) ; j<=k ; j++)
	      ELEMENT(im,i,j)=col;
	    k = (int) floor(iny+coefdir*(i-inx+1));
	    for (j=(int) floor(iny+coefdir*(i-inx+0.5)) ; j<=k ; j++)
	      ELEMENT(im,i+1,j)=col;
	  }
      else
	for (i=inx;i<fnx;i++)
	  {
	    k = (int) ceil(iny+coefdir*(i-inx+0.5));
	    for (j=(int) ceil(iny+coefdir*(i-inx)) ; j>=k ; j--)
	      ELEMENT(im,i,j)=col;
	    k = (int) ceil(iny+coefdir*(i-inx+1));
	    for (j=(int) ceil(iny+coefdir*(i-inx+0.5)) ; j>=k ; j--)
	      ELEMENT(im,i+1,j)=col;
	  }
    }

  return(EXIT_SUCCESS);
}

int fill_triangle(image im, int centre_x, int centre_y, int P1x, int P1y, int P2x, int P2y, int P3x, int P3y, color col)
{
  register int i,j;
  register int box_LR_x,box_LR_y;
  register int x1,x2,x3,y1,y2,y3;
  int box_UL_x,box_UL_y;

  x1 = centre_x+P1x;
  x2 = centre_x+P2x;
  x3 = centre_x+P3x;
  y1 = centre_y+P1y;
  y2 = centre_y+P2y;
  y3 = centre_y+P3y;

  box_UL_x = (x2>x1) ?  x1 : x2;
  box_UL_x = (box_UL_x>x3) ?  x3 : box_UL_x;

  box_LR_x = (x2>x1) ?  x2 : x1;
  box_LR_x = (box_LR_x<x3) ?  x3 : box_LR_x;
  
  box_UL_y = (y2>y1) ?  y1 : y2;
  box_UL_y = (box_UL_y>y3) ?  y3 : box_UL_y;

  box_LR_y = (y2>y1) ?  y2 : y1;
  box_LR_y = (box_LR_y<y3) ?  y3 : box_LR_y;

  if ( (x1<0) || (x1>=im.nx) || (x2<0) || (x2>=im.nx) || (x3<0) || (x3>=im.nx) || (y1<0) || (y1>=im.ny) || (y2<0) || (y2>=im.ny) || (y3<0) || (y3>=im.ny) )
    return (EXIT_FAILURE);

  if ( (x1==x2 && y1==y2) || (x1==x3 && y1==y3) || (x3==x2 && y3==y2) )
    return (EXIT_FAILURE);

  for (j=box_UL_y;j<=box_LR_y;j++)
    for (i=box_UL_x;i<=box_LR_x;i++)
      	if ( ( ((i-x2)*(y3-y2) - (j-y2)*(x3-x2)) * ((x1-x2)*(y3-y2)-(y1-y2)*(x3-x2)) >=0) && (((i-x1)*(y3-y1) - (j-y1)*(x3-x1)) * ((x2-x1)*(y3-y1)-(y2-y1)*(x3-x1)) >=0) && ( ((i-x1)*(y2-y1) - (j-y1)*(x2-x1)) * ((x3-x1)*(y2-y1)-(y3-y1)*(x2-x1))  >=0))
	  ELEMENT(im,i,j) = col;
      

  return (EXIT_SUCCESS);

}

int MakeColorImage(colormap *cm, image im, int nx, int ny, int fmt_dbl, double *dataD, float *data, double min, double max)
{
  /* This function fills an image with a field colored according to the colormap provided. The fmt_dbl flags says if the datas are contained in the double or float array. min is the value corresponding to the low extrema of the colormap and max is the other end.*/
  register int i,j,position=0,index;


  if ( (im.nx!=nx) || (im.ny!=ny) || isnull(im) || (min>max) )
    return(EXIT_FAILURE);

  
  if (fmt_dbl==0)
    {
      for (j=0;j<ny;j++)
	for (i=0;i<nx;i++,position++)
	  {
	    index = (int) floor((cm->nb_entries-1)*(data[position]-min)/(max-min));
	    ELEMENT(im,i,j) = (cm->rgb)[index];
	  }
    }
  else
    {
      for (j=0;j<ny;j++)
	for (i=0;i<nx;i++,position++)
	  {
	    index = (int) floor((cm->nb_entries-1)*(dataD[position]-min)/(max-min));
	    ELEMENT(im,i,j) = (cm->rgb)[index];
	  }
    }
  
  return (EXIT_SUCCESS);
  
}

int WritePPM(FILE *output, image im)
{
  int i,nx,ny;
  if (isnull(im))
    return (EXIT_FAILURE);

  nx=im.nx;
  ny=im.ny;

  fprintf(output,"P6\n#\n%d %d\n255\n",nx,ny);

  for (i=0;i<ny;i++)
    fwrite(ADRESSE(im,0,i),sizeof(color),nx,output);

  return (EXIT_SUCCESS);
}

int ReadPPM(FILE *input, image im)
{
  int i,nx,ny;
  char buffer[BUFSIZ];
  int total_read;
  if (isnull(im))
    return (EXIT_FAILURE);

  do
    {
      fgets(buffer,BUFSIZ,input);
      if (feof(input))
	return(EXIT_FAILURE);
    }
  while (strncmp(buffer,"P6",2));
  do
    fgets(buffer,BUFSIZ,input);
  while (buffer[0]=='#');
  sscanf(buffer,"%d %d\n",&nx,&ny);
  fgets(buffer,BUFSIZ,input);

  if ((nx!=im.nx)||(ny!=im.ny))
    return(EXIT_FAILURE);

  total_read=0;
  for (i=0;i<ny;i++)
    total_read+=fread(ADRESSE(im,0,i),sizeof(color),nx,input);

  if (total_read!=nx*ny)
    return (EXIT_FAILURE);

  return (EXIT_SUCCESS);
}

image CreatePPMFromFile(FILE *input)
{
  int i,nx,ny;
  image result;
  char buffer[BUFSIZ];
  int total_read;

  fgets(buffer,BUFSIZ,input);
  if (strncmp(buffer,"P6",2))
      return(NULL_image);
  do
    fgets(buffer,BUFSIZ,input);
  while (buffer[0]=='#');
  sscanf(buffer,"%d %d\n",&nx,&ny);

  fgets(buffer,BUFSIZ,input);

  result.image = (color*) malloc (nx*ny*sizeof(color));

  total_read=0;
  for (i=0;i<ny;i++)
    total_read+=fread(result.image,sizeof(color),nx,input);
  
  if (total_read!=nx*ny)
    return (NULL_image);

  result.nx=nx;
  result.ny=ny;
  result.saut=nx;
  
  return (result);
} 

image CreatePPMFromPBMFile(FILE *input)
{
  int i,nx,ny;
  image result;
  char buffer[BUFSIZ];
  int total_read;

  fgets(buffer,BUFSIZ,input);
  if (strncmp(buffer,"P4",2))
      return(NULL_image);
  do
    buffer[0]=fgetc(input);
  while (buffer[0]=='#');
  sscanf(buffer,"%d %d\n",&nx,&ny);
  fgetc(input);

  result.image = (color*) malloc (nx*ny*sizeof(color));

  total_read=0;
  for (i=0;i<ny;i++)
    total_read+=fread(result.image,sizeof(color),nx,input);
  
  if (total_read!=nx*ny)
    return (NULL_image);
  
  return (result);
} 

void SizePPMFile(FILE *input,int *nx, int *ny)
{
  char buffer[BUFSIZ];
  fgets(buffer,BUFSIZ,input);

  do
    fgets(buffer,BUFSIZ,input);
  while (buffer[0]=='#');

  sscanf(buffer,"%d %d",nx,ny);
  rewind(input);
}

int Add_scale_bar(image im,int n,int n_inter,color col)
{
  register int i,j;
  int size_x_main,size_y_main,size_x_inter,size_y_inter,l;

  size_y_main = im.ny;
  size_x_main = (int) floor(size_y_main*1.15470053838);
 
  size_y_inter = (size_y_main) / 2;
  size_x_inter = (int) floor(size_y_inter*1.15470053838); 

  l = im.nx - size_x_main -2;
  
  for (j=0;j<n;j++)
    {
      fill_triangle(im,size_x_main/2+(j*l)/n,size_y_main-1,-size_x_main/2,-size_y_main+1,0,0,size_x_main/2,-size_y_main+1,col);
      for (i=1;i<n_inter;i++)
	fill_triangle(im,size_x_main/2+(j*l)/n + (i*l)/(n_inter*n),size_y_inter-1,-size_x_inter/2,-size_y_inter+1,0,0,size_x_inter/2,-size_y_inter+1,col);
    }
  fill_triangle(im,size_x_main/2+(j*l)/n,size_y_main-1,-size_x_main/2,-size_y_main+1,0,0,size_x_main/2,-size_y_main+1,col);
  
  return(size_x_main);

}

int Add_Amplitude_bar(image im, double gmin, double gmax, double min,double max)
{
  int s,scale_thikness=4,nmin,nmax,light_width=5;
  double maximum;
  image Warn_light_min;
  image Warn_light_max;
  image Centre,values;
  image scale_bar_up;

  if ((im.nx<70) || (im.ny<12))
    return(EXIT_FAILURE);
  
  maximum = 2*((gmin*gmin>gmax*gmax) ? ((gmin<0) ? -gmin : gmin ) : ((gmax<0) ? -gmax : gmax ));

  scale_bar_up=sub_image(im,light_width+3,0,im.nx-2*(light_width+3),scale_thikness);
  s=Add_scale_bar(scale_bar_up,4,5,color_WHITE);

  nmin = (int) ceil((scale_bar_up.nx-s-1)*min/maximum)+1;
  nmax = (int) floor((scale_bar_up.nx-s-1)*max/maximum)-1;

  nmin = (nmin < -((scale_bar_up.nx-s-1)/2)-1) ? -((scale_bar_up.nx-s)/2-1)  : nmin ;
  nmax = (nmax > (scale_bar_up.nx-s-1)/2 ) ? ((scale_bar_up.nx-s-1)/2)  : nmax ;
  
  values = sub_image(im,im.nx/2+nmin-1,scale_thikness+1,nmax-nmin+1,im.ny-scale_thikness-1);
  fill_image(values,color_GRAY);

  Centre = sub_image(im,im.nx/2-2,scale_thikness+1,3,im.ny-scale_thikness-1);

  Warn_light_min = sub_image(im,im.nx/2+nmin-light_width-2,scale_thikness+1,light_width,values.ny);
  Warn_light_max = sub_image(im,im.nx/2+nmax+1,scale_thikness+1,light_width,values.ny);
  
  if (gmin>min)
    fill_image(Warn_light_min,color_RED);

  if (max>gmax)
    fill_image(Warn_light_max,color_RED);
  
  fill_image(Centre,color_BLUE);  

  return(EXIT_SUCCESS);
}

int Add_Colorbar(image im, colormap *cm, int vertical)
{

  register int i,j,imax,jmax,nb;
  imax = im.nx;
  jmax = im.ny;
  nb=cm->nb_entries - 1;
  /* Fill an image with the colorbar corresponding to the colormap cm */

  if (isnull(im) || (cm==NULL))
    return (EXIT_FAILURE);
  
  if (vertical==1)
    for (j=0;j<jmax;j++)
      for (i=0;i<imax;i++)
	ELEMENT(im,i,j) = (cm->rgb)[(j*nb)/(jmax-1)];
  else
    for (j=0;j<jmax;j++)
      for (i=0;i<imax;i++)
	ELEMENT(im,i,j) = (cm->rgb)[(i*nb)/(imax-1)];
      
  return (EXIT_SUCCESS);
}

color MakeColor(unsigned char Red, unsigned char Green, unsigned char Blue)
{
  color out;
  out.red=Red;
  out.green=Green;
  out.blue=Blue;
  return(out);
}

color SetColorIntensity(color C,float intensity)
{
  color out;
  unsigned char max;
  intensity = COLOR_VALUE_RANGE * ((intensity>1) ? 1 : ( (intensity>=0) ? intensity : 0));
  max = C.blue;
  if (C.green>max)
    max=C.green;
  if (C.red>max)
    max=C.red;
  out.blue=(unsigned char) ((intensity * C.blue)/max);
  out.green=(unsigned char) ((intensity * C.green)/max);
  out.red=(unsigned char) ((intensity * C.red)/max);
  return(out);
}
