% Heat_Optimization.m: Script for reconstructing optimal Neumann boundary 
% condition from data for the 1D heat equation (PM)
% Includes Sobolev Gradients, Conjugate Gradients (with resets), and 
% a constraint imposed on the flux!
% Requires file Heat_RHS.m to produce "True_Function.mat"

% Written by Pritpal Matharu	2018/10/25
% McMaster University
%

function Heat_Optimization
clear all
close all
clc
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
datetime
tic
% Use Conjugate gradient method or Steepest descent method
% Set as 0 to use the standard steepest descent method
% Otherwise use the Conjugate gradient method
CG_method = 1;

%% Whether or not the additional linear constraint is added to the model
% Set as 1 to add the additional linear constraint
% Otherwise use original problem
flux_constr = 1;

%% Load variables for problem
load('True_Function.mat') 
% u_b .......... Reference time-history of RHS boundary
% phi_true ..... True Neumann boundary condition for comparison
% dx ........... Step size in the x domain
% dt ........... Step size in the time domain
% x ............ Spatial domain, discretized equispaced (0 <= x <= 1)
% t ............ Time domain, discretized equispaced    (0 <= t <= 1)
% x_len ........ Reference length for x array
% t_len ........ Reference length for time array
% h ............ Magnitude reference
% u0 ........... Initial condition
% A ............ Matrix for solving PDES. Crank-Nicolson method is used. 

%% Optimizing Parameters
% Tolerance for optimization
tol = 1e-5;
% maximum number of iterations
max_iter = 10000;
% Length scale for solving Sobolev gradient, which ensures suitable regularization
L = 1.0;
%L = 0.01;
% Starting step length for line search
tau0 = 100;
% Bracketing Factor, to enlarge & shrink the fminbnd interval
brack_fact = 2;
% Clearing frequency is dependent whether or not we use Conjugate Gradients
switch CG_method
    case 0 % Standard steepest descent method
        % If using the standard steepest descent method, we simply clear
        % the gradient every iteration
        fq = 1;
    otherwise % Conjugate gradient
        % Assign value of clearing frequency of PR beta value
        fq = 10;
end

% Initial Guess for Neumann Boundary Function 
phi(1, :) = 18*t*exp(-4);

%% ORIGINAL FUNCTIONAL - Calculate values for initial guess
% Calculate Cost Functional
J1 = costfun(phi, u0, A, dx, dt, u_b, t); % Value of initial cost function
J2 = 0;                                   % Dummy value

% Store values for initial guess
Tau(1, 1)  = tau0;
J(1, 1)    = J1;

% Initialization iteration counter
p = 1;

%% Loop Iteration
% Iterate until condition is met
while ( (abs(J1 - J2))/(abs(J1)) > tol & p <= max_iter )
    
    %% Determine Sobolev Gradient
    % Solve Heat Equation (forward in time)
    [u_r, ~] = heatsolve(phi, u0, A, dx, dt);
    
    % Solve Adjoint system and determine L2 gradient (backwards in time)
    [grad_L2] = gradsolve(u_r, A, dx, dt, u_b, t_len);
        
    % Determine Sobolev gradient with or without linear constraint on flux
    switch flux_constr
        case 1 % Add additional linear constraint on Neumann BC

            % Determine Sobolev gradient, for additional regularity
            % Also solve with additional linear constraint on flux
            [del_J] = Sobolevgrad(grad_L2, dt, t_len, L, flux_constr);
        otherwise % Determine regular Sobolev gradient
            
            % Determine Sobolev gradient, for additional regularity
            [del_J] = Sobolevgrad(grad_L2, dt, t_len, L);
    end
    
    %% Conjugate Gradient method
    % Use Conjugate gradient method, with the Polak-Ribiere method 
    % including frequency resetting
    if p >= 2 && mod(p, fq)~=0
        % Using the Polak Ribere Method
        bPR = ((del_J)*((del_J - delk)'))/(norm(delk)^2);
        
        % Use value to create the conjugate gradient
        PR = del_J + bPR.*pr;
    else
        % Frequency clearing to reset the conjugate-gradient procedure
        bPR = 0; % Save for diagnostics
        
        % Use gradient (previous gradient cleared)
        PR = del_J;
        
        % Display the current iteration (based on frequency clearing)
        fprintf('Iteration: %i, Cost Function: %d \n', p, J1)
    end
    
    % Ensure that the cost does not equal zero (only for the first iteration)
    if p ~= 1
        % Set Cost functional equal to current iteration
        J1 = J2;
    end
    
    % Bracketing the minimum from the right, for the Brent method
    Feval = 1;        % Count number of function calls
    Gtau  = [0.0 J1]; % Store step length and respective value of cost function
    
    % Calculate the cost function for the given step length and gradient
    J2 = costfun(phi + tau0*PR, u0, A, dx, dt, u_b, t);
    
    %% Modifying the interval for step lengths
    if J2 > J1
        % If the Cost Functional at tau0 is GREATER than the current iteration of
        % the evaluated Cost Functional, we try to shrink the bracket interval that
        % we evaluate the fminbnd function
        
        % Shrink bracket interval, until appropriate value is found
        while J2 > J1
            % Shrink by a constant bracketing factor
            tau0 = tau0/brack_fact;

            % Calculate the cost function for the given step length and gradient
            J2 = costfun(phi + tau0*PR, u0, A, dx, dt, u_b, t);
            
            % Update the number of times the function is called
            Feval = Feval + 1;
            % Update store the step length used, and the value of cost function
            Gtau = [Gtau; [tau0, J2] ];
        end
        % To ensure that we did not shrink the bracket too much!
        tau0 = tau0*brack_fact;
        
    else
        % If the Cost Functional at tau0 is less than the current iteration of the
        % evaluated Cost Functional, we try to expand the bracket that we evaluate
        % the fminbnd function at because there may exist an even larger interval
        % that could give us an even smaller (and better) step length for the Cost
        % Functional!
        
        % Expand bracket interval, until appropriate value is found
        while J2 < J1
            % Expand by a constant bracketing factor
            tau0 = tau0*brack_fact;
            
            % Calculate the cost function for the given step length and gradient
            J2 = costfun(phi + tau0*PR, u0, A, dx, dt, u_b, t);
            
            % Update the number of times the function is called
            Feval = Feval + 1;
            % Update store the step length used, and the value of cost function
            Gtau = [Gtau; [tau0, J2] ];
        end
    end % End of if statement
    
    %% Determine optimal value for steplength
    % Determine step length along the gradient, using line search method
    [tau0, J2, Gtau, exitflag, output] = costfunmin(phi, PR, u0, A, dx, dt, u_b, t, tau0, Gtau);
    
    Feval = Feval + length(Gtau);
    
    % Diagnostics for minimization process
    if ( exitflag ~= 1 )
        fprintf('  PROBLEM: exitflag=%d \n', exitflag);
    end
    
    %% Storing Values
    % For diagnostics, save results from line search
    Exit(p, 1) = exitflag;
    Out(p, :)  = output;
    g{p, 1}    = Gtau;
    Num(p, 1)  = Feval;
    CG(p, :)   = PR;
    bval(p, :) = bPR;

    % Store values for the Polak Ribere method
    delk = del_J;
    pr   = PR;
    
    % Store values
    grad(p, :)   = del_J;
    ur(p, :)     = u_r;
    Phi(p, :)    = phi;
    J(p+1, 1)    = J2;
    Tau(p+1, 1)  = tau0;
    
    %%  Update Neumann Boundary condition
    phi = phi + tau0*PR;

    % Increment iteration counter
    p = p+1;
       
end

% Solve Heat Equation (forward in time) 
[u_r, ~] = heatsolve(phi, u0, A, dx, dt);

% Storing optimal values
ur(p, :)   = u_r; % Right boundary time history
Phi(p, :)  = phi; % Neumann boundary condition

% Stop timing
time = toc;

% Display Optimization information
fprintf('\n Number of iterations: %i, Final Value: %i, Time: %i \n', p, J(end), time)

%% Plots 
str = sprintf('Optimization_Summary.mat');

% Right Time history
iter1 = floor(p/25);
iter2 = floor(p/10);
iter3 = floor(p/2);
figure()
plot(t, u_b, '-k')
hold on
ax = gca;
ax.ColorOrderIndex = 1;
plot(t, ur(1, :), '-.')
ax.ColorOrderIndex = 3;
plot(t, ur(iter1+1, :), ':')
plot(t, ur(iter2+1, :), '.-')
plot(t, ur(iter3+1, :), '.-.')
ax.ColorOrderIndex = 2;
plot(t, u_r(1, :), '--')
hold off
xlabel('$t$', 'Interpreter','LaTex')
ylabel('$u(\phi(t))|_{x=b}$', 'Interpreter','LaTex')
Legend{1}     = sprintf('True Function');
Legend{end+1} = sprintf('Iteration 0');
Legend{end+1} = sprintf('Iteration %i', iter1);
Legend{end+1} = sprintf('Iteration %i', iter2);
Legend{end+1} = sprintf('Iteration %i', iter3);
Legend{end+1} = sprintf('Optimal Reconstruction');
legend(Legend, 'Interpreter', 'LaTex', 'location', 'best')
title('Temperature of the Right Endpoint', 'Interpreter','LaTex')

% Left Neumann boundary condition
figure()
plot(t, phi_true, '-k')
hold on
ax = gca;
ax.ColorOrderIndex = 1;
plot(t, Phi(1, :), '-.')
ax.ColorOrderIndex = 3;
plot(t, Phi(iter1+1, :), ':')
plot(t, Phi(iter2+1, :), '.-')
plot(t, Phi(iter3+1, :), '.-.')
ax.ColorOrderIndex = 2;
plot(t, phi(1, :), '--')
xlabel('$t$', 'Interpreter','LaTex')
ylabel('$\phi(t)$', 'Interpreter','LaTex')
legend(Legend, 'Interpreter', 'LaTex', 'location', 'best')
title('Left Neumann Boundary Condition', 'Interpreter','LaTex')

% Cost function
figure()
semilogy(0:length(J)-1, J, '--')
xlabel('$n$', 'Interpreter', 'LaTex')
ylabel('$\mathcal{J}(\phi^{(n)})$', 'Interpreter', 'LaTex')
str4 = sprintf('Cost Functional');
title({str4}, 'Interpreter','LaTex')

% Line search
figure()
semilogy(g{1}(:, 1), g{1}(:, 2), 'k.-', 'MarkerSize', 10.0)
hold on
ax = gca;
ax.ColorOrderIndex = 1;
semilogy(g{iter1}(:, 1), g{iter1}(:, 2), '.-', 'MarkerSize', 10.0)
ax.ColorOrderIndex = 3;
semilogy(g{iter2}(:, 1), g{iter2}(:, 2), '.-', 'MarkerSize', 10.0)
semilogy(g{iter3}(:, 1), g{iter3}(:, 2), '.-', 'MarkerSize', 10.0)
ax.ColorOrderIndex = 2;
semilogy(g{end}(:, 1), g{end}(:, 2), '.-', 'MarkerSize', 10.0)
hold off
xlabel('$\tau$', 'Interpreter','LaTex')
ylabel('$\mathcal{J}(\phi^{(n)} - \tau \nabla_{\phi}\mathcal{J}(\phi^{(n)}))$', 'Interpreter','LaTex')
Legend1{1}     = sprintf('Iteration 1');
Legend1{end+1} = sprintf('Iteration %i', iter1);
Legend1{end+1} = sprintf('Iteration %i', iter2);
Legend1{end+1} = sprintf('Iteration %i', iter3);
Legend1{end+1} = sprintf('Final Iteration');
legend(Legend1, 'Interpreter', 'LaTex', 'location', 'best')
title('Line Search Iterations', 'Interpreter','LaTex')


% Save information
save(str, 'p', 'time', 'J', 'Tau', 'Phi', 'ur', 'grad', 'g', 'Exit', 'Out', 'Num', 'CG', 'L', 'bval', 'fq')

end % End of function

% -------------------------------------------------------------------------
% FUNCTION: heatsolve
%
% AUTHOR ... Pritpal Matharu
% DATE ..... 2018/10/12
%
% Solves the Heat Equation and the outputs the right side values
%
% INPUT
% phi ....... Current Neumann Boundary condition
% u0 ........ Initial Condition Function
% A ......... Matrix for next time step
% dx ........ Step size in the spatial domain
% dt ........ Step size in the time domain
%
% OUTPUT
% u_r ....... Heat Equation solved at right limit
% u ......... Solution to Heat Equation
%
% FORMAT
% [u_r, u] = heatsolve(phi, u0, A, dx, dt)
%
% -------------------------------------------------------------------------
function [u_r, u] = heatsolve(phi, u0, A, dx, dt)

% Reference lengths
x_len = length(u0);  % Reference length for spatial dimension
t_len = length(phi); % Reference length for time dimension
h     = dt/(dx^2);   % Magnitude reference

% Storage Matrices
u     = zeros(x_len, t_len);
y_PDE = zeros(x_len, 1);

% Apply Initial Condition at t=0
u(:, 1) = u0;

% Solve Heat equation system for each time step forwards
for n = 1:t_len-1
    % Solve time step forwards
    y_PDE(1)         = 3*u(1, n) - 4*u(2, n) + u(3, n) + 2*dx*(phi(n) + phi(n+1));
    y_PDE(2:x_len-1) = 0.5*h*u(1:x_len-2, n) + u(2:x_len-1, n) - h*u(2:x_len-1, n) + 0.5*h*u(3:x_len, n);
    y_PDE(x_len)     = u(x_len-2, n) - 4*u(x_len-1, n) + 3*u(x_len, n);
    
    % Solve next time step
    u(:, n+1) = A \ y_PDE;
end

% Heat equation solved at right limit
u_r = u(end, :);

end % End of function

% -------------------------------------------------------------------------
% FUNCTION: gradsolve
%
% AUTHOR ... Pritpal Matharu
% DATE ..... 2018/10/12
%
% Solves the Adjoint System and determines the L2 gradient
% 
% INPUT
% u_r ....... Heat Equation solved at right limit
% A ......... Matrix for next time step
% dx ........ Step size in the spatial domain
% dt ........ Step size in the time domain
% u_b ....... Ideal function 
% t_len...... Reference length for time dimension
%
% OUTPUT
% del_J ..... L2 Gradient of system
%
% FORMAT
% [grad_L2] = gradsolve(u_r, A, dx, dt, u_b, t_len)
%
% -------------------------------------------------------------------------
function [grad_L2] = gradsolve(u_r, A, dx, dt, u_b, t_len)

% Reference lengths
x_len = length(A); % Reference length for spatial dimension
h     = dt/(dx^2); % Magnitude reference

% Storage Matrices
v     = zeros(x_len, t_len);
y_adj = zeros(x_len, 1);

% Solve adjoint system for each time step BACKWARDS
for m = t_len:-1:2
    % Solve time step backwards
    y_adj(1)         = 3*v(1, m) - 4*v(2, m) + v(3, m);
    y_adj(2:x_len-1) = 0.5*h*v(1:x_len-2, m) + v(2:x_len-1, m) - h*v(2:x_len-1, m) + 0.5*h*v(3:x_len, m);
    y_adj(x_len)     = 3*v(x_len, m) - 4*v(x_len-1, m) + v(x_len-2, m) - 2*dx*((u_r(m-1) + u_r(m)) - (u_b(m-1) + u_b(m)));
    
    % Solve previous time step
    v(:, m-1) = A \ y_adj;
end

% L2 Gradient of system
grad_L2 = (-1)*(v(1, :));

end % End of function

% -------------------------------------------------------------------------
% FUNCTION: Sobolevgrad
%
% AUTHOR ... Pritpal Matharu
% DATE ..... 2018/10/25
%
% Solves Sobolev (H^1) gradient of the Heat Equation, given the L^2
% gradient
% If additional flag is activated, will also ensure the gradient has zero
% integral over the time interval (Neumann BC has constant flux)
%
% INPUT
% grad_L2 ....... L2 gradient, solved using adjoint analysis
% dt ............ Step size in the time domain
% t_len ......... Reference length for time dimension
% L ............. Sobolev parameter, to ensure regularity
% ~ ............. Dummy entry, as a flag to activate the linear constraint
%
% OUTPUT
% del_J ..... Sobolev Gradient of system
%
% FORMAT
% [del_J] = Sobolevgrad(grad_L2, dt, t_len, l, ~)
%
% -------------------------------------------------------------------------
function [del_J] = Sobolevgrad(grad_L2, dt, t_len, L, ~)

% Storage vector
del_J = zeros(1, t_len);

% Create Matrix, in order to solve the PDE resulting from the Sobolev
% Gradient System
P = sparse(1:t_len-3, 2:t_len-2, ((-1)*((L/dt)^2)), t_len-2, t_len-2); % Upper Diagonal Entries
Q = sparse(1:t_len-2, 1:t_len-2, (2*((L/dt)^2) + 1),t_len-2, t_len-2); % Diagonal Entries
R = sparse(2:t_len-2, 1:t_len-3, ((-1)*((L/dt)^2)), t_len-2, t_len-2); % Lower Diagonal Entries
A = P+Q+R;

% Determine whether additional linear constraint is activated
switch nargin
    case 5 % Impose additional constraint on gradient
        % Modify matrix to include integral constraint
        A(1:t_len-2, t_len-1) = 1.0;    % Last Column (Lagrange multiplier)
        A(t_len-1, 1:t_len-2) = 1.0*dt; % Interior weights for trapz

        % Modified RHS, adding zero since we want the resulting gradient to have zero integral 
        grad_Lin = [grad_L2(2:end-1)'; 0];
        
        % Solve gradient with linear constraint
        grad_H1  = A \ grad_Lin;

        % Gradient of system (negative value, to go in the DESCENT direction)
        % Note: We don't need the value of the linear constraint, so we don't include it
        del_J(2:end-1) = -grad_H1(1:end-1);

    otherwise % Determine regular Sobolev gradient
        % Solve the gradient for the middle terms
        grad_H1 = A \ (grad_L2(2:end-1))';
        
        % Gradient of system (negative value, to go in the DESCENT direction)
        del_J(2:end-1) = -grad_H1;
end

end % End of function

% -------------------------------------------------------------------------
% FUNCTION: costfun
%
% AUTHOR ... Pritpal Matharu
% DATE ..... 2018/10/12
%
% Calculates the Cost Functional for the given Heat Equation system 
%
% USES THE RIGHT BOUNDARY
%
% INPUT
% phi ....... Current Neumann Boundary condition
% u0 ........ Initial Condition Function
% A ......... Matrix for next time step
% dx ........ Step size in the spatial domain
% dt ........ Step size in the time domain
% u_b ....... Ideal function (Right boundary)
% t ......... Time domain
%
% OUTPUT
% J ......... Cost Functional
% u_r ....... Heat Equation solved at right limit (optional)
% FORMAT
% [J, u_r] = costfun(phi, u0, A, dx, dt, u_b, t)
% -------------------------------------------------------------------------
function [J, u_r] = costfun(phi, u0, A, dx, dt, u_b, t)

% Heat equation solved at right limit
[u_r, ~] = heatsolve(phi, u0, A, dx, dt);

% Calculate Cost Functional 
J = (1/2)*trapz(t, ((u_r - u_b).^2));

end

% -------------------------------------------------------------------------
% FUNCTION: costfunmin
%
% AUTHOR ... Pritpal Matharu
% DATE ..... 2018/10/22
%
% Minimizing the cost functional, to obtain the optimal step length to take 
%
% INPUT
% phi ....... Current Neumann Boundary condition
% grad ...... Conjugate gradient
% u0 ........ Initial Condition Function
% A ......... Matrix for next time step
% dx ........ Step size in the spatial domain
% dt ........ Step size in the time domain
% u_b ....... Ideal function (Right boundary)
% t ......... Time domain
%
% MAINTAIN A RECORD OF VALUES
% tau ....... Initial step length
% history ... History, containing step length, and value of cost functional
%
% OUTPUT
% tau ....... Optimal step length
% fval ...... Value of cost functional, corresponding to optimal tau
% history ... History, containing step length, and value of cost functional
% exitflag .. Exitflag, to ensure the minimization performed correctly
% output .... Diagnostic, of the minimization process
%
% FORMAT
% [tau, fval, history, exitflag, output] = costfunmin(phi, grad, u0, A, dx, dt, u_b, t, tau, history)
% -------------------------------------------------------------------------
function [tau, fval, history, exitflag, output] = costfunmin(phi, grad, u0, A, dx, dt, u_b, t, tau, history)

% Set options, to output the history
options = optimset('OutputFcn', @myoutput);

% The cost functional is line-minimized with respect to 's'
[tau, fval, exitflag, output] = fminbnd(@(s) costfun(phi + s*grad, u0, A, dx, dt, u_b, t), 0.0, tau, options);

% Nested function, to output the history
    function stop = myoutput(x,optimvalues,state)
        % Flag for exiting function
        stop = false;
        % If still iterating, update history vector
        if isequal(state,'iter')
            % Append the step-size and corresponding value to history 
            history = [history; [x, optimvalues.fval] ];
        end
    end
end
