function grad_v1
close all, clear all

%set(gca,'FontSize',20);

% PARAMETERS
% variant of CG: 1 --- Fletcher-Reeves, 2 --- Polak-Ribiere
CG = 2;

% CG: frequency of restarts
CGrestart = 10;

% the initial guess 
x0 = [-1.9; 2.0];

% minimizer
xt = [1.0; 1.0];

% maximum number of iterations
ITERMAX = 10000;

% (fixed) step size
dt = 1.0e-3;

% tolerance for gradient iterations
epsf = 1.0e-6;

% tolerance for line search
epst = 1.0e-5;

% shrink/stretch factor for bracketing in line search
tau_brack_fac  = 2.0;

% maximum number of function evaluations in line search
MAX_LS_EVALS = 30;

CONSTRAINT = 0; 
% definition of the constraint
xc = linspace(-2.0,2.0,100);
yc = -0.05*xc.^4 + 2.651605;
if ( CONSTRAINT == 1 )
  dt = 1.0e-4;    % smaller time step required for the contrained problem
end


% First, visualize the banana function
Nx = 100;
x1 = linspace(-2,2,Nx);
x2 = linspace(-1,3,Nx); 

[X1,X2] = meshgrid(x1,x2);
for i = 1:Nx
  for j = 1:Nx
    F(i,j) = Banana( [x1(j), x2(i)] );
  end;
end;


figure(1)
surf( X1, X2, F, 'edgecolor','none' )
xlabel('x_1','FontSize', 18), ylabel('x_2','FontSize', 18), zlabel('f(x_1,x_2)','FontSize', 18)
colormap(flipud(autumn))
title('banana function','FontSize', 18)


figure(2)
contourf( X1, X2, F, 50,'LineColor','none')
colormap(flipud(autumn))
xlabel('x_1','FontSize', 18), ylabel('x_2','FontSize', 18);
colorbar
title('banana function','FontSize', 18)
if ( CONSTRAINT == 1 )
  hold on
  plot(xc,yc,'-k');
end

fprintf('\nTo continue press any key ... \n')
pause

% =================================================================================

% STEEPEST DESCENT METHOD WITH FIXED STEP SIZE
fprintf('==== Steepest Descent method with fixed step size ==== \n')

n = 0;

x1 = x0;
f1 = Banana( x1 );
f2 = 0.5 * f1; % just to get into the loop
xn = x1';

SDdiag = [n, x1', f1, norm(x1 - xt) ];
fprintf(' n=%3d  x1=%10.8e  x2=%10.8e  f=%10.8e \n',...
	n, x1(1), x1(2), f1)

while ( (abs( f1 - f2 ) / f1) > epsf & ( n < ITERMAX ) )

% evaluate the gradient 
  gradf = Df( x1 );
  
  if ( CONSTRAINT == 1 )
% gradient of the constraint function
    gradc = [ -0.2*x1(1)^3; -1.0 ];
    
% determine the projection operator
    Ps = eye(2) - gradc * gradc' / norm(gradc)^2;
    
% determine the projected gradient 
    gradf = Ps * gradf;
  end;
  
% update 
  x2 = x1 - dt * gradf;
  
% bookkeeping
  xn = [xn; x2'];
  x1 = x2;
  f1 = f2;
  f2 =  Banana( x2 );
    
  n = n + 1;
  
% save diagnostics
  SDdiag = [SDdiag; [n, x2', f2, norm(x2 - xt) ] ];

  if ( mod(n,100) == 0 )
    fprintf(' n=%3d  x1=%10.8e  x2=%10.8e  f=%10.8e \n',...
	    n, x2(1), x2(2), f2)
  end;
  
end;

figure(12)
contour( X1, X2, F, 50)
%colormap(flipud(autumn))
xlabel('x_1','FontSize', 18), ylabel('x_2','FontSize', 18);
title('Steepest Descent')
hold on
plot(xn(:,1),xn(:,2),'-xb')
if ( CONSTRAINT == 1 )
  plot(xc,yc,'-m');
end
axis([-2.0 2.0 -1.0 3.0]);

figure(61)
semilogy(SDdiag(:,1),SDdiag(:,4),'-xb')
xlabel('n','FontSize', 18), ylabel('f(x^n)','FontSize', 18)
legend('SD')

figure(62)
semilogy(SDdiag(:,1),SDdiag(:,5),'-xb')
xlabel('n','FontSize', 18), ylabel('$||x^n - \tilde{x}||$','Interpreter','latex','FontSize', 18)
legend('SD')

figure(63), 
Nsd = min(1000,size(SDdiag,1));
semilogy(SDdiag(1:Nsd,1),dt,'-xb'), hold on
xlabel('n','FontSize', 18), ylabel('$\tau_n$','Interpreter','latex','FontSize', 18)
legend('SD')



if ( CONSTRAINT == 1 )
  return;
end

fprintf('\nTo continue press any key ... \n')
pause


% =================================================================================

% STEEPEST DESCENT METHOD WITH LINE SEARCH

fprintf('\n\n\n\n\n')
fprintf('==== Steepest Descent method with line search ==== \n')

opts_linemin = optimset('Display','notify','TolX',epst,'MaxIter',MAX_LS_EVALS);

n = 0;

t0 = dt;
x1 = x0;
f1 = Banana( x1 );
f2 = 0.5 * f1; % just to get into the loop
xn = x1';

SDLSdiag = [n, x1', f1, norm(x1 - xt), t0, 0 ];
fprintf(' n=%3d  x1=%10.8e  x2=%10.8e  f=%10.8e \n',...
	n, x1(1), x1(2), f1)

while ( (abs( f1 - f2 ) / f1) > epsf & ( n < ITERMAX ) )

% evaluate the gradient 
  gradf = Df( x1 );
  
% line mininization;  
% bracketing the minimum from the right;
    
  if ( Phi(t0,x1,-gradf) > f1 )
% try to shrink the bracket
    while (  Phi(t0,x1,-gradf) > f1 )
      t0 = t0 / tau_brack_fac;
    end
    t0 = t0 * tau_brack_fac;
  else
    while ( Phi(t0,x1,-gradf) < f1 )
      t0 = t0 * tau_brack_fac;
    end
  end
	
  [tn,fn,exitflag,output] = fminbnd(@(t) Phi(t,x1,-gradf),0.0,t0,opts_linemin);
  
  if ( exitflag ~= 1 )
    fprintf('    PROBLEMS:  exitflag=%d \n', exitflag );
  end;
 	
% update 
  x2 = x1 - tn * gradf;
  
% bookkeeping
  xn = [xn; x2'];
  x1 = x2;
  f1 = f2;
  f2 =  Banana( x2 );
    
  n = n + 1;
  
% save diagnostics
  SDLSdiag = [SDLSdiag; [n, x2', f2, norm(x2 - xt), tn, output.funcCount] ];

  if ( mod(n,100) == 0 )
    fprintf(' n=%3d  x1=%10.8e  x2=%10.8e  f=%10.8e \n',...
	    n, x2(1), x2(2), f2)
  end;
end;

figure(13)
contour( X1, X2, F, 50)
%colormap(flipud(autumn))
xlabel('x_1','FontSize', 18), ylabel('x_2','FontSize', 18);
title('Steepest Descent with Line Search')
hold on
plot(xn(:,1),xn(:,2),'-or')


figure(61), hold on
semilogy(SDLSdiag(:,1),SDLSdiag(:,4),'-or')
legend('SD','SDLS')


figure(62), hold on
semilogy(SDLSdiag(:,1),SDLSdiag(:,5),'-or')
legend('SD','SDLS')

figure(63), hold on
semilogy(SDLSdiag(1:1000,1),SDLSdiag(1:1000,6),'-or')
legend('SD','SDLS')

fprintf('\nTo continue press any key ... \n')
pause

% =================================================================================

% CONJUGATE GRADIENTS METHOD WITH LINE SEARCH

fprintf('\n\n\n\n\n')
fprintf('==== Conjugate Gradients method with line search ==== \n')

opts_linemin = optimset('Display','notify','TolX',epst,'MaxIter',MAX_LS_EVALS);

n = 0;

t0 = dt;
x1 = x0;
f1 = Banana( x1 );
f2 = 0.5 * f1; % just to get into the loop
x2 = x1;
gradf = Df( x1 );
pn = gradf;
xn = x1';

CGdiag = [n, x1', f1, norm(x1 - xt), t0, 0 ];
fprintf(' n=%3d  x1=%10.8e  x2=%10.8e  f=%10.8e \n',...
	n, x1(1), x1(2), f1)

while ( (abs( f1 - f2 ) / f1) > epsf & ( n < ITERMAX ) ...
	& (norm(x2 - xt) > 1.0e-13) )

% evaluate the gradient 
  gradf2 = gradf;
  gradf = Df( x1 );
  
% evaluate the momentum term
  if ( CG == 1 )
% Fretcher-Reeves
    beta = (gradf' * gradf) / (gradf2' * gradf2);
  elseif ( CG == 2 )
% Polak-Ribiere
    beta = (gradf' * (gradf - gradf2)) / (gradf2' * gradf2);
  else
% steepest descent
    beta = 0.0;
  end;
    
  if ( mod(n,CGrestart) == 0 )
    beta = 0.0;
  end;
  
  
% construct the descent direction
  pn2 = pn;
  if ( n > 0 )
    pn = -gradf + beta*pn2;
  else
    pn = -gradf;
  end;
  
% line mininization;  
% bracketing the minimum from the right;
    
  if ( Phi(t0,x1,pn) > f1 )
% try to shrink the bracket
    while (  Phi(t0,x1,pn) > f1 )
      t0 = t0 / tau_brack_fac;
    end
    t0 = t0 * tau_brack_fac;
  else
    while ( Phi(t0,x1,pn) < f1 )
      t0 = t0 * tau_brack_fac;
    end
  end
	
  [tn,fn,exitflag,output] = fminbnd(@(t) Phi(t,x1,pn),0.0,t0,opts_linemin);
  
  if ( exitflag ~= 1 )
    fprintf('    PROBLEMS:  exitflag=%d \n', exitflag );
  end;
 	
% update 
  x2 = x1 + tn * pn;
  
% bookkeeping
  xn = [xn; x2'];
  x1 = x2;
  f1 = f2;
  f2 =  Banana( x2 );
    
  n = n + 1;
  
% save diagnostics
  CGdiag = [CGdiag; [n, x2', f2, norm(x2 - xt), tn, output.funcCount] ];

  if ( mod(n,1) == 0 )
    fprintf(' n=%3d  x1=%10.8e  x2=%10.8e  f=%10.8e \n',...
	    n, x2(1), x2(2), f2)
  end;

end;

figure(14)
contour( X1, X2, F, 50)
%colormap(flipud(autumn))
xlabel('x_1','FontSize', 18), ylabel('x_2','FontSize', 18);
title('Conjugate Gradients with Line Search')
hold on
plot(xn(:,1),xn(:,2),'-sk')


figure(61), hold on
semilogy(CGdiag(:,1),CGdiag(:,4),'-sk')
legend('SD','SDLS','CG')


figure(62), hold on
semilogy(CGdiag(:,1),CGdiag(:,5),'-sk')
legend('SD','SDLS','CG')

figure(63), hold on
semilogy(CGdiag(:,1),CGdiag(:,6),'-sk')
legend('SD','SDLS','CG')



% =================================================================================
% Functions
% =================================================================================

% Rosenbrock's "banana function": 
function [ f ] = Banana( x )

f = (100*(x(2) - x(1).^2).^2 + (1 - x(1)).^2);

% =================================================================================

% gradient of the "banana function": 
function [ gradf ] = Df( x )
  
  gradf = [-400*(x(2) - x(1)^2)*x(1) - 2*(1 - x(1));
	200*(x(2) - x(1)^2)];

% =================================================================================

function [ ft ] = Phi( t, x, pn )
  
  ft = Banana(x + t * pn);

