% KappaTest.m: Script of the Kappa Test for the one dimensional Heat
% Equation (PM)
% Tests the L2 gradient, since the Sobolev gradient is simply a filtered
% version of the L2 gradient

% Written by Pritpal Matharu	2018/10/19
% McMaster University
%


function KappaTest
clear all
close all
clc
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
datetime
tic

%% Set varying step sizes
d_x = [0.1, 0.01, 0.001];
d_t = [0.1, 0.01, 0.001];

% Various linestyles for plotting
line_sty = {'.-.', '.-', '--'};

% Loop through each step size
for m = 1:length(d_t)
for n = 1:length(d_x)
    % Ensure certain variables do not get cleared
    clearvars -except d_x d_t m n ax ax1 line_sty Legend
    
    %% Initial Setup
    % Step Sizes
    dx = d_x(n); % Step size in the x domain for current iteration
    dt = d_t(m); % Step size in the time domain for current iteration
    
    % Boundaries
    x_min = 0;  % Left Boundary Point  (x = a)
    x_max = 1;  % Right Boundary Point (x = b)
    t_i   = 0;  % Starting time        (t = 0)
    t_f   = 1;  % Final time           (t = T)
    
    % Create arrays
    x = x_min:dx:x_max; % Create x array, with equally spaced intervals
    t = t_i:dt:t_f;     % Create time array, with equally spaced intervals
    
    % Reference lengths
    x_len = length(x); % Reference for x array
    t_len = length(t); % Reference for time array
    h     = dt/(dx^2); % Magnitude reference
    
    %% Functions from PDE
    % Initial Condition
    u0  = cos(pi*x);
    
    % Initial Guess for Neumann Boundary Function
    phi(1, :) = 4*t*exp(-4);
        
    %% Matrix Setup
    % Determine values in matrix for PDE solvers. The matrix A is constant.
    A_U = sparse(1:x_len-1, 2:x_len, (-0.5*h),x_len, x_len); % Upper Diagonal Entries
    A_D = sparse(1:x_len, 1:x_len, (1 + h),x_len, x_len);    % Main  Diagonal Entries
    A_L = sparse(2:x_len, 1:x_len-1, (-0.5*h),x_len, x_len); % Lower Diagonal Entries
    A = A_U + A_D + A_L;
    A(1, 1) = -3; % Left Boundary Condition, First Row
    A(1, 2) = 4;  % Left Boundary Condition, First Row
    A(1, 3) = -1; % Left Boundary Condition, First Row
    A(x_len, x_len)     = -3; % Right Boundary Condition, Last Row
    A(x_len, x_len - 1) = 4;  % Right Boundary Condition, Last Row
    A(x_len, x_len - 2) = -1; % Right Boundary Condition, Last Row
    
    %% Kappa Test parameters
    % Range of epsilon for kappa test
    ep = logspace(-12, 1, 14);
    
    % Storage Matrix
    kappa = zeros(1, length(ep));
    
    % Perturbation function
    phi_prime = 4*(t).*exp(-4*t + pi*t);
    
    %% ORIGINAL FUNCTIONAL
    % Solve Heat Equation and Gradient system for each time step forwards
    % using the true solution for the cost function
    phi_true = 4*t.*exp(-4*t);
    [u_b, ~] = heatsolve(phi_true, u0, A, dx, dt);
    
    % Solve Heat Equation and cost function, with no perturbation
    [J, u_r] = costfun(phi, u0, A, dx, dt, u_b, t);
    
    % Solve Adjoint system and determine L2 gradient (backwards in time)
    [del_J]  = gradsolve(u_r, A, dx, dt, u_b, t_len);
    
    % Denominator of the Kappa Test
    kap      = trapz(t, del_J.*phi_prime);
    
    %% Perturbation Loop
    for p = 1:length(ep)
        % Solve Heat Equation with Perturbation
        [u_R, ~] = heatsolve(phi+ep(p)*phi_prime, u0, A, dx, dt);
                
        % Calculate Cost Functional with Perturbation
        J_per    = (1/2)*(trapz(t, (u_R - u_b).^2));
        
        % Solve for numerator of Kappa Test
        kappa(p) = ((J_per - J))/(kap*ep(p));
    end
    
    %% Plotting
    % Initialize plotting
    if n == 1 && m == 1
        % Kappa Test
        figure(1)
        clf;
        ax = gca;
        semilogx(ep, kappa, line_sty{m})
        hold on

        % Log-based difference Kappa test
        figure(2)
        clf;
        ax1 = gca;
        kappa_log = log10(abs(kappa - 1));
        semilogx(ep, kappa_log, line_sty{m})
        hold on

        % First legend entry
        Legend{1} = sprintf('$\\Delta x= %.1d, \\Delta t= %.1d$', dx, dt);
    else
        % Kappa Test
        figure(1)
        semilogx(ep, kappa, line_sty{m})
        
        % Log-based difference Kappa test
        figure(2)
        kappa_log = log10(abs(kappa - 1));
        semilogx(ep, kappa_log, line_sty{m})
        
        % Update legend entry
        Legend{end+1}     = sprintf('$\\Delta x= %.1d, \\Delta t= %.1d$', dx, dt);
    end
    
end
% Reset colouring scheme for plotting
ax.ColorOrderIndex  = 1;
ax1.ColorOrderIndex = 1;
end

%% Finalize Plots
% Kappa test plot
figure(1)
hold off
ylim([0 2])
xlabel('$\epsilon$', 'Interpreter', 'LaTex')
ylabel('$\kappa(\epsilon)$', 'Interpreter', 'LaTex')
legend(Legend, 'Interpreter', 'LaTex', 'location', 'best')
str2 = sprintf('Kappa Test');
title({str2}, 'Interpreter','LaTex')

%  Log-based difference Kappa Test
figure(2)
hold off
xlabel('$\epsilon$', 'Interpreter', 'LaTex')
ylabel('$\log_{10}|1 - \kappa(\epsilon)|$', 'Interpreter', 'LaTex')
legend(Legend, 'Interpreter', 'LaTex', 'location', 'best')
str2 = sprintf('Kappa Test (logarithmic scaling)');
title({str2}, 'Interpreter','LaTex')

toc
end % End of function


% -------------------------------------------------------------------------
% FUNCTION: heatsolve
%
% AUTHOR ... Pritpal Matharu
% DATE ..... 2018/10/12
%
% Solves the Heat Equation and the outputs the right side values
%
% INPUT
% phi ....... Current Neumann Boundary condition
% u0 ........ Initial Condition Function
% A ......... Matrix for next time step
% dx ........ Step size in the spatial domain
% dt ........ Step size in the time domain
%
% OUTPUT
% u_r ....... Heat Equation solved at right limit
% u ......... Solution to Heat Equation
%
% FORMAT
% [u_r, u] = heatsolve(phi, u0, A, dx, dt)
%
% -------------------------------------------------------------------------
function [u_r, u] = heatsolve(phi, u0, A, dx, dt)

% Reference lengths
x_len = length(u0);  % Reference length for spatial dimension
t_len = length(phi); % Reference length for time dimension
h     = dt/(dx^2);   % Magnitude reference

% Storage Matrices
u     = zeros(x_len, t_len);
y_PDE = zeros(x_len, 1);

% Apply Initial Condition at t=0
u(:, 1) = u0;

% Solve Heat equation system for each time step forwards
for n = 1:t_len-1
    % Solve time step forwards
    y_PDE(1)         = 3*u(1, n) - 4*u(2, n) + u(3, n) + 2*dx*(phi(n) + phi(n+1));
    y_PDE(2:x_len-1) = 0.5*h*u(1:x_len-2, n) + u(2:x_len-1, n) - h*u(2:x_len-1, n) + 0.5*h*u(3:x_len, n);
    y_PDE(x_len)     = u(x_len-2, n) - 4*u(x_len-1, n) + 3*u(x_len, n);
    
    % Solve next time step
    u(:, n+1) = A \ y_PDE;
end

% Heat equation solved at right limit
u_r = u(end, :);

end % End of function

% -------------------------------------------------------------------------
% FUNCTION: gradsolve
%
% AUTHOR ... Pritpal Matharu
% DATE ..... 2018/10/12
%
% Solves the Adjoint System and determines the L2 gradient
% 
% INPUT
% u_r ....... Heat Equation solved at right limit
% A ......... Matrix for next time step
% dx ........ Step size in the spatial domain
% dt ........ Step size in the time domain
% u_b ....... Ideal function 
% t_len...... Reference length for time dimension
%
% OUTPUT
% del_J ..... L2 Gradient of system
%
% FORMAT
% [grad_L2] = gradsolve(u_r, A, dx, dt, u_b, t_len)
%
% -------------------------------------------------------------------------
function [grad_L2] = gradsolve(u_r, A, dx, dt, u_b, t_len)

% Reference lengths
x_len = length(A); % Reference length for spatial dimension
h     = dt/(dx^2); % Magnitude reference

% Storage Matrices
v     = zeros(x_len, t_len);
y_adj = zeros(x_len, 1);

% Solve adjoint system for each time step BACKWARDS
for m = t_len:-1:2
    % Solve time step backwards
    y_adj(1)         = 3*v(1, m) - 4*v(2, m) + v(3, m);
    y_adj(2:x_len-1) = 0.5*h*v(1:x_len-2, m) + v(2:x_len-1, m) - h*v(2:x_len-1, m) + 0.5*h*v(3:x_len, m);
    y_adj(x_len)     = 3*v(x_len, m) - 4*v(x_len-1, m) + v(x_len-2, m) - 2*dx*((u_r(m-1) + u_r(m)) - (u_b(m-1) + u_b(m)));
    
    % Solve previous time step
    v(:, m-1) = A \ y_adj;
end

% L2 Gradient of system
grad_L2 = (-1)*(v(1, :));

end % End of function

% -------------------------------------------------------------------------
% FUNCTION: costfun
%
% AUTHOR ... Pritpal Matharu
% DATE ..... 2018/10/12
%
% Calculates the Cost Functional for the given Heat Equation system 
%
% USES THE RIGHT BOUNDARY
%
% INPUT
% phi ....... Current Neumann Boundary condition
% u0 ........ Initial Condition Function
% A ......... Matrix for next time step
% dx ........ Step size in the spatial domain
% dt ........ Step size in the time domain
% u_b ....... Ideal function (Right boundary)
% t ......... Time domain
%
% OUTPUT
% J ......... Cost Functional
% u_r ....... Heat Equation solved at right limit (optional)
% FORMAT
% [J, u_r] = costfun(phi, u0, A, dx, dt, u_b, t)
% -------------------------------------------------------------------------
function [J, u_r] = costfun(phi, u0, A, dx, dt, u_b, t)

% Heat equation solved at right limit
[u_r, ~] = heatsolve(phi, u0, A, dx, dt);

% Calculate Cost Functional 
J = (1/2)*trapz(t, ((u_r - u_b).^2));

end
