% Heat_Single.m: Script for solving 1D heat equation on the right side (PM)
% 
% This script is just to produce a value for the right side of the boundary
% condition, to set as the ideal function (u_b). This will allow the
% optimization functions to be tested to ensure that it can produce the
% same results. When we use this ideal function, we will pretend that the
% Initial Neumann Function Guess (phi) is unknown, and also see how close
% phi is produced to the actual phi value (which is given here in the
% script)!
% 

% Written by Pritpal Matharu	2018/10/12
% McMaster University
%
%
function Heat_RHS
clear all
close all
clc
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Initial Setup
% Step Sizes
dx = 0.01;  % Step size in the x domain
dt = 0.01;  % Step size in the time domain

% Boundaries
x_min = 0;  % Left Boundary Point  (x = a)
x_max = 1;  % Right Boundary Point (x = b)
t_i   = 0;  % Starting time        (t = 0)
t_f   = 1;  % Final time           (t = T)

% Create arrays
x = x_min:dx:x_max; % Create x array, with equally spaced intervals
t = t_i:dt:t_f;     % Create time array, with equally spaced intervals

% Reference lengths
x_len = length(x); % Reference for x array
t_len = length(t); % Reference for time array
h     = dt/(dx^2); % Magnitude reference

%% Functions from PDE
% Initial Condition
u0  = cos(pi*x);   

% True Neumann boundary function
phi = 18*t.*exp(-4*t);  % ***Solve the system according to this function!***

%% Matrix Setup (Crank-Nicolson method)
% Determine values in matrix for PDE solvers. The matrix A is constant.
A_U = sparse(1:x_len-1, 2:x_len, (-0.5*h),x_len, x_len); % Upper Diagonal Entries
A_D = sparse(1:x_len, 1:x_len, (1 + h),x_len, x_len);    % Main  Diagonal Entries
A_L = sparse(2:x_len, 1:x_len-1, (-0.5*h),x_len, x_len); % Lower Diagonal Entries
A = A_U + A_D + A_L;
A(1, 1) = -3; % Left Boundary Condition, First Row
A(1, 2) = 4;  % Left Boundary Condition, First Row
A(1, 3) = -1; % Left Boundary Condition, First Row
A(x_len, x_len)     = -3; % Right Boundary Condition, Last Row
A(x_len, x_len - 1) = 4;  % Right Boundary Condition, Last Row
A(x_len, x_len - 2) = -1; % Right Boundary Condition, Last Row

%% ORIGINAL FUNCTIONAL
% Solve Heat Equation for each time step forwards
[u_b, u] = heatsolve(phi, u0, A, dx, dt);

% Save true boundary
phi_true = phi;

% Surface plot of the evolution of the heat equation
figure()
surf( t, x, u );
ylabel('$x$', 'Interpreter', 'LaTex')
xlabel('$t$', 'Interpreter', 'LaTex')
zlabel('$u(x, t)$', 'Interpreter', 'LaTex')
set(gca,'Ydir','reverse')
title('Heat Equation Evolution', 'Interpreter','LaTex')

% Target Function
figure()
plot(t, u_b, '-k')
xlabel('$t$', 'Interpreter','LaTex')
ylabel('$u(\phi(t))|_{x=b}$', 'Interpreter','LaTex')
title('Temperature of the Right Endpoint (Target Function)', 'Interpreter','LaTex')

% Save RHS and parameters for the problem 
str = 'True_Function';
save(str, 'phi_true', 'u_b', 'dx', 'dt', 'x', 't', 'x_len', 't_len', 'h', 'u0', 'A')

end % End of function

% -------------------------------------------------------------------------
% FUNCTION: heatsolve
%
% AUTHOR ... Pritpal Matharu
% DATE ..... 2018/10/12
%
% Solves the Heat Equation and the outputs the right side values
%
% INPUT
% phi ....... Current Neumann Boundary condition
% u0 ........ Initial Condition Function
% A ......... Matrix for next time step
% dx ........ Step size in the spatial domain
% dt ........ Step size in the time domain
%
% OUTPUT
% u_r ....... Heat Equation solved at right limit
% u ......... Solution to Heat Equation
%
% FORMAT
% [u_r, u] = heatsolve(phi, u0, A, dx, dt)
%
% -------------------------------------------------------------------------
function [u_r, u] = heatsolve(phi, u0, A, dx, dt)

% Reference lengths
x_len = length(u0);  % Reference length for spatial dimension
t_len = length(phi); % Reference length for time dimension
h     = dt/(dx^2);   % Magnitude reference

% Storage Matrices
u     = zeros(x_len, t_len);
y_PDE = zeros(x_len, 1);

% Apply Initial Condition at t=0
u(:, 1) = u0;

% Solve Heat equation system for each time step forwards
for n = 1:t_len-1
    % Solve time step forwards
    y_PDE(1)         = 3*u(1, n) - 4*u(2, n) + u(3, n) + 2*dx*(phi(n) + phi(n+1));
    y_PDE(2:x_len-1) = 0.5*h*u(1:x_len-2, n) + u(2:x_len-1, n) - h*u(2:x_len-1, n) + 0.5*h*u(3:x_len, n);
    y_PDE(x_len)     = u(x_len-2, n) - 4*u(x_len-1, n) + 3*u(x_len, n);
    
    % Solve next time step
    u(:, n+1) = A \ y_PDE;
end

% Heat equation solved at right limit
u_r = u(end, :);

end % End of function